/*
 * EventDB.h
 *
 * Copyright (C) 1999 Karim Yaghmour.
 *
 * This is distributed under GPL.
 *
 * Header for event database engine for trace toolkit.
 *
 * History : 
 *    K.Y., 26/06/99, Initial typing.
 */

#ifndef __TRACE_TOOLKIT_EVENT_DB__
#define __TRACE_TOOLKIT_EVENT_DB__

#include "TraceToolkit.h"
#include <stdio.h>

/* Definitions */
#define EVENT_STRING_MAX_SIZE       100
#define HEADER_BANER  "####################################################################\n"
#define HEADER_CPUID  "CPU-ID"
#define HEADER_EVENT  "Event"
#define HEADER_TIME   "\t\tTime(sec, usec)\t"
#define HEADER_PID    "\tPID"
#define HEADER_LENGTH "\tLength"
#define HEADER_STRING "\tDescription"

/******************************************************************
 * Macro :
 *    PrintData()
 * Description :
 *    Prints out the given data in the given format to the given file.
 * Parameters :
 *    pmFile, File to which we write.
 *    pmFormat, Format of the data.
 *    pmArg, The data to be printed
 * History :
 *    K.Y., 17/10/99, Initial typing
 ******************************************************************/
#define DBPrintData(pmFile, pmFormat, pmArg...) \
do \
{\
  char         lWriteString[100];\
\
  sprintf(lWriteString, pmFormat, ##pmArg);\
  write(pmFile, lWriteString, strlen(lWriteString));\
} while(0)

/* Time operation macros */
/*  (T3 = T2 - T1) */
#define DBTimeSub(T3, T2, T1) \
do \
{\
  T3.tv_sec  = T2.tv_sec  - T1.tv_sec;  \
  T3.tv_usec = T2.tv_usec - T1.tv_usec; \
  if(T3.tv_usec < 0)\
    {\
    T3.tv_sec--;\
    T3.tv_usec += 1000000;\
    }\
} while(0)

/*  (T3 = T2 + T1) */
#define DBTimeAdd(T3, T2, T1) \
do \
{\
  T3.tv_sec  = T2.tv_sec  + T1.tv_sec;  \
  T3.tv_usec = T2.tv_usec + T1.tv_usec; \
  if(T3.tv_usec > 1000000)\
    {\
    T3.tv_sec += T3.tv_usec / 1000000;\
    T3.tv_usec = T3.tv_usec % 1000000;\
    }\
} while(0)

/*  Get the time in the double format */
#define DBGetTimeInDouble(X) (double) (((double)X.tv_sec)*1000000 + (double)X.tv_usec)

/* Events are equal */
#define DBEventsEqual(X, Y)   ((X.BufID == Y.BufID) && (X.EventPos == Y.EventPos))

#define DBIEventGetBufStart(pmDB, pmEvent) \
        (pmEvent->BufID * pmDB->BufferSize + pmDB->TraceStart)
#define DBIEventGetPosition(pmDB, pmEvent) \
        (DBIEventGetBufStart(pmDB, pmEvent) + pmEvent->EventPos)

/* Event structure */
typedef struct _event
{
  uint32_t         BufID;    /* Buffer ID of the buffer to which this event belongs */
  uint32_t         EventPos; /* Event position in the buffer */
} event;

typedef struct _eventDescription
{
  /* Characteristics as read from raw trace */
  int              ID;                             /* The ID number of the event as given in linux/include/linux/profile.h */
  struct timeval   Time;                           /* Time at which event occured */
  int              CPUID;                          /* CPU on which this event occured */
  void*            Struct;                         /* Structured describing the event */
  char             String[EVENT_STRING_MAX_SIZE];  /* String describing the event */
  int              Size;                           /* Size of entry describing event (as was recorded in kernel trace module) */
} eventDescription;

/* Event database */
typedef struct _db
{
  void*                  TraceStart;          /* Address at which the raw trace starts */
  off_t                  TraceSize;           /* Size of the trace */
  uint32_t               BufferSize;          /* Size of buffers */

  trace_event_mask       EventMask;           /* The event mask */
  trace_event_mask       DetailsMask;         /* Are the event details logged */
  uint8_t                LogCPUID;            /* Is the CPUID logged */

  unsigned long int      Nb;                  /* Number of events */
  event                  FirstEvent;          /* First event in database */
  event                  LastEvent;           /* Last event in event list */
  struct timeval         StartTime;           /* Time at which trace started */
  struct timeval         EndTime;             /* Time at which trace ended */

#if GTK_ENV
  event                  FirstEventWithProc;  /* First event that belongs to a process */
  struct timeval         DrawStartTime;       /* Time at which we should start drawing */
#endif
} db;

/* Syscall tracking structure */
typedef struct _syscall
{
  int              ID;    /* The ID number of the system call */
  int              Nb;    /* Number of times system call was called */
  int              Depth; /* Depth of system call */
  struct timeval   Time;  /* Total amount of time passed in system call */
  struct _syscall* Next;  /* Next system call */
} syscall;

/* Process description */
typedef struct _process
{
  /* Properties */
  int               PID;           /* PID of process */
  int               PPID;          /* PID of parent */
  char*             Command;       /* Command given to start process */
  int               NbSyscalls;    /* Number of system calls that occured */
  int               NbTraps;       /* Number of traps that occured */
  int               QFileRead;     /* Quantity of data read by process to files */
  int               QFileWrite;    /* Quantity of data written by process to files */
  struct timeval    TimeOfBirth;   /* Time at which process was created */
  struct timeval    TimeProcCode;  /* Time during which we were executing process code */
  struct timeval    TimeRuning;    /* Time during which this process had the control over the CPU */
  struct timeval    TimeIO;        /* Time spent waiting for I/O */

  /* Variables used during analysis */
  int               Depth;         /* Current syscall depth */
  syscall*          Pending;       /* Syscalls not completely analyzed yet */
  int               CtrlDepth;     /* Depth in control events */
  event             LastIOEvent;   /* Last I/O event */
  event             LastSchedIn;   /* Last time the task was scheduled in */

  /* Related structures */
  syscall*          Syscalls;      /* Process specific syscall analysis  */
  struct _system*   System;        /* System to which this process belongs */

  /* List hooks */
  struct _process*  Next;          /* Next process in list */
  struct _process*  Children;      /* Pointer to first child */
  struct _process*  NextChild;     /* Next child in child list */ 
  struct _process*  NextHash;      /* Next process in hash table entry */
#if GTK_ENV
  /* Graphic properties of process */
  int               ListPos;       /* Position of process in process list */
#endif /* GTK_ENV */
} process;

typedef uint8_t proc_hash;
#define PROC_HASH_SIZE   256

/* System database */
typedef struct _system
{
  /* Statistics */
  /*  Core events */
  unsigned long int  SyscallEntry; /* Number of system call entries */
  unsigned long int  SyscallExit;  /* Number of system call exits */
  unsigned long int  TrapEntry;    /* Number of trap entries */
  unsigned long int  TrapExit;     /* Number of trap exits */
  unsigned long int  IRQEntry;     /* Number of interrupt entries */
  unsigned long int  IRQExit;      /* Number of interrupt exits */
  unsigned long int  SchedChange;  /* Number of scheduler changes */
  unsigned long int  KernelTimer;  /* Number of times the kernel timer has been called */
  unsigned long int  BH;           /* Number of bottom-halves that occured */

  /*  Extend events set */
  unsigned long int  TimerExpire;  /* Number of timer expiries */
  unsigned long int  PageAlloc;    /* Number of page allocations */
  unsigned long int  PageFree;     /* Number of page frees */
  unsigned long int  PacketOut;    /* Number of outgoing packets */
  unsigned long int  PacketIn;     /* Number of incoming packets */

  /* Lists of system components */
  process*           Processes;                  /* All the processes seen during trace */
  process*           ProcTree;                   /* The process tree */
  process            ProcHash[PROC_HASH_SIZE];   /* Hash table containing pointer to the processes */
  syscall*           Syscalls;                   /* System-wide syscall analysis */  
} system;

/* The trace manipulating functions */
uint8_t DBEventID(db* pmDB, event* pmEvent);

void DBEventTime(db* pmDB, event* pmEvent, struct timeval* pmTime);

uint8_t DBEventCPUID(db* pmDB, event* pmEvent);

void* DBEventStruct(db* pmDB, event* pmEvent);

int DBEventString(db* pmDB, event* pmEvent, char* pmString, int pmStringLen);

uint8_t DBEventSize(db* pmDB, event* pmEvent);

int DBEventControl(db* pmDB, event* pmEvent, system* pmSystem);

process* DBEventProcess(db* pmDB, event* pmEvent, system* pmSystem);

void DBEventDescription(db* pmDB, event* pmEvent, int pmStringRequired, eventDescription* pmEventDescription);

int DBEventNext(db* pmDB, event* pmEvent);

int DBEventPrev(db* pmDB, event* pmEvent);


int      DBReadTrace
          (int       /* File from which data is to be read */,
	   db*       /* Database to be filled */);
void     DBProcessProcInfo
          (FILE*     /* File containing proc info */,
	   system*   /* System description to be filled */);
process* DBGetProcByPID
          (int       /* PID of process to find */,
           system*   /* System in which process is to be found */);
process* DBFindProcInTree
          (int       /* PID of process to find */,
           process*  /* Process tree in which to search */);
void     DBProcessTrace
          (system*   /* System to be processed */,
	   db*       /* Database of events */,
	   options*  /* The processing options */);
void     DBPrintTrace
          (int       /* File to which data is to be printed */,
	   system*   /* System who's informationt is to be printed */,
	   db*       /* Database of events */,
	   options*  /* Options */);
process* DBCreateProcess
          (system*   /* The system to which the process belongs */,
           int       /* The PID of the created process */,
           int       /* The PID of the parent process */);
db*      DBCreateDB
          (void);
system*  DBCreateSystem
          (void);
void     DBDestroyTrace
          (system*   /* The system to be destroyed */,
           db*       /* The event database to be destroyed */);


#endif /* __TRACE_TOOLKIT_EVENT_DB__ */
