/*
   EventGraph.c : Event graph engine.
   Copyright (C) 1999, 2000 Karim Yaghmour.
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program;- if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

   History : 
      K.Y., 10/02/2000, EGIDrawEvents() has undergone another major rewrite
                        because of the changes in the way the trace is read.
                        The event structure not holding the same information
                        it had, details are fetched dynamically using the
                        corresponding DBEventXXX() functions.
      K.Y., 20/10/1999, EGIDrawEvents() has gone a major rewrite. Now, it uses
                        IRQ entries and exits and has been corrected for some
                        nuissances and complexities it has. It is still complex
                        though, but there's no way around this if Linux' behavior
                        is to be presented as it really is.
      K.Y., 19/07/1999, Adding detail to the graph after separating EGIDrawTrace()
                        in smaller pieces.
      K.Y., 07/07/1999, Using a pixmap the size of drawable area. Initially
                        drawing was done using a "true" size pixmap containing
			all the trace. This took too much memory.
      K.Y., 28/06/1999, Initial typing.
*/

#include <math.h>             /* This is for fmod */
#include "TraceToolkit.h"
#include "Tables.h"
#include "EventDB.h"
#include "MainWindow.h"
#include "EventGraph.h"

const gchar*  gProcessItemData = "Process Item Data";

/**********************************************************************************/
/******************************* Internal functions *******************************/
/**********************************************************************************/
/******************************************************************
 * Function :
 * Description :
 * Parameters :
 * Return values :
 * History :
 * Note :
 ******************************************************************/
#if 1
#define EGIFindEventDistance(EGraph, EventDesc, DrawStart, DrawEnd) \
   (guint) ((((gdouble) EGraph->HDrawSize) * \
                (DBGetTimeInDouble(EventDesc.Time) - DrawStart)) / (DrawEnd - DrawStart))
#else /* This is for debugging only */
guint EGIFindEventDistance(eventGraph* pmEventGraph,
			   eventDescription pmEventDesc, 
			   gdouble pmDrawStart,
			   gdouble pmDrawEnd)
{
  guint lRetValue;
  gdouble lDistance;

  lDistance = (DBGetTimeInDouble(pmEventDesc.Time) - pmDrawStart) * ((gdouble) pmEventGraph->HDrawSize);
  lDistance /= pmDrawEnd - pmDrawStart;

  g_print("lDistance = %lf \n", lDistance);

  lRetValue = (guint) lDistance;

  g_print("lRetValue = %d \n", lRetValue);

  return lRetValue;
}
#endif

/******************************************************************
 * Function :
 * Description :
 * Parameters :
 * Return values :
 * History :
 * Note :
 ******************************************************************/
#define EGIFindProcessHeight(EGraph, Process) \
   EGraph->VTrueSize - \
   (EGraph->NbProc - Process->ListPos) * EGraph->LabelHeight + \
   EGraph->LabelHeight / 3

/******************************************************************
 * Function :
 * Description :
 * Parameters :
 * Return values :
 * History :
 * Note :
 ******************************************************************/
#define EGIFindKernelHeight(EGraph) \
   EGraph->VTrueSize - EGraph->LabelHeight + EGraph->LabelHeight / 3

/******************************************************************
 * Function :
 * Description :
 * Parameters :
 * Return values :
 * History :
 * Note :
 ******************************************************************/
#define	EGIDrawThickHLine(PixMap, GC, x1, y1, x2, y2) \
   gdk_draw_line(PixMap, GC, x1, y1, x2, y2); \
   gdk_draw_line(PixMap, GC, x1, y1 + 1, x2, y2 + 1); \
   gdk_draw_line(PixMap, GC, x1, y1 + 2, x2, y2 + 2);

/******************************************************************
 * Function :
 * Description :
 * Parameters :
 * Return values :
 * History :
 * Note :
 ******************************************************************/
GdkColor* EGICreateColor(long pmRed, long pmGreen, long pmBlue)
{
  GdkColor*   pColor;     /* The color */

  /* Allocate space for color */
  pColor = (GdkColor*) g_malloc(sizeof(GdkColor));

  /* Set the color's attributes */
  pColor->red   = pmRed;
  pColor->green = pmGreen;
  pColor->blue  = pmBlue;

  /* Allocate the color in the colormap */
  gdk_color_alloc(gdk_colormap_get_system(), pColor);

  /* Give the color to the caller */
  return pColor;
}

/******************************************************************
 * Function :
 * Description :
 * Parameters :
 * Return values :
 * History :
 * Note :
 ******************************************************************/
GdkGC* EGICreateGC(GdkPixmap*   pmPixMap,
		   GdkColor*    pmForegroundColor,
		   GdkColor*    pmBackgroundColor,
		   GdkLineStyle pmLineType)
{
  GdkGC*  pGC;   /* The graphic context */

  /* Allocate space for GC */
  pGC = gdk_gc_new(pmPixMap);

  /* Set the forground and background colors */
  gdk_gc_set_foreground(pGC, pmForegroundColor);
  gdk_gc_set_background(pGC, pmBackgroundColor);

  /* Set the Line Type */
  gdk_gc_set_line_attributes(pGC,
			     1,
			     pmLineType,
			     GDK_CAP_NOT_LAST,
			     GDK_JOIN_MITER);


  /* Return the graphic contex to the caller */
  return pGC;
}

/******************************************************************
 * Function :
 * Description :
 * Parameters :
 * Return values :
 * History :
 * Note :
 ******************************************************************/
void  EGIFindDrawLimits(eventGraph* pmEventGraph,
			gfloat   pmValue,     gfloat   pmPageSize,
			gfloat   pmMinValue,  gfloat   pmMaxValue, 
			gdouble* pmDrawStart, gdouble* pmDrawEnd)
{
  /* Are we drawing the last page */
  if(pmValue > (pmMaxValue - pmPageSize))
    {
    /* Set drawing times */
    *pmDrawStart = pmEventGraph->StartTime + pmEventGraph->Duration - pmEventGraph->Interval;
    *pmDrawEnd   = pmEventGraph->StartTime + pmEventGraph->Duration;
    }
  else if(pmValue == pmMinValue)
    {
    /* Set drawing times */
    *pmDrawStart = pmEventGraph->StartTime;
    *pmDrawEnd   = pmEventGraph->StartTime + pmEventGraph->Interval;
    }
  else
    {
    /* Set drawing times */
    *pmDrawStart = pmEventGraph->StartTime +
      (pmEventGraph->Duration *
      (gdouble) ((pmValue - pmMinValue)) / (gdouble) (pmMaxValue - pmMinValue));
    *pmDrawEnd   = *pmDrawStart + pmEventGraph->Interval;
    }
}

/******************************************************************
 * Function : EGIFindLimitEvents()
 * Description :
 *    Find the events at the limits of the draw area.
 * Parameters :
 *    pmEventGraph, The event graph for which the search is carried
 *                  out
 *    pmDrawStart, The time at which we start drawing.
 *    pmDrawEnd, The time at which we end drawing.
 *    pmFirstEvent, The pointer to the first event to be drawn.
 *    pmLastEvent, The pointer to the last event to be drawn.
 * Return values :
 *    TRUE, We found drawing limits.
 *    FALSE, We failed to find draw limits.
 * History :
 * Note :
 ******************************************************************/
int EGIFindLimitEvents(eventGraph* pmEventGraph,
		       gdouble     pmDrawStart,  gdouble pmDrawEnd,
		       event*      pmFirstEvent, event*  pmLastEvent)
{
  event              lEvent;               /* Generic event pointer */
  event              lEventNext;           /* Next event */
  event              lEventPrev;           /* Previous event */
  event              lFirstEvent = {0, 0}; /* First event drawn */
  event              lLastEvent = {0, 0};  /* Last event drawn */
  struct timeval     lEventTime;           /* An event time */
  struct timeval     lEventNextTime;       /* Next event's time */
  struct timeval     lEventPrevTime;       /* Previous event's time */

  /* Were we dealing with the first event in the event list */
  if(pmDrawStart == pmEventGraph->StartTime)
    lFirstEvent = pmEventGraph->EventDB->FirstEventWithProc;
  else
    {
    /* Are we starting at the same time as the already draw event */
    if(pmDrawStart == pmEventGraph->DrawStartTime)
      /* Start at the same event as last time */
      lFirstEvent = pmEventGraph->DrawStartEvent;
    else
      {
      /* Start at the current draw start event */
      lEventPrev = lEventNext = lEvent = pmEventGraph->DrawStartEvent;

      /* Are we moving forward */
      if(pmDrawStart >= pmEventGraph->DrawStartTime)
	{
	/* Keep going on forward in the trace */
	while(DBEventNext(pmEventGraph->EventDB, &lEventNext) == TRUE)
	  {
	  /* Get the event time */
	  DBEventTime(pmEventGraph->EventDB, &lEvent, &lEventTime);
	  DBEventTime(pmEventGraph->EventDB, &lEventNext, &lEventNextTime);

	  /* Is the next event past the draw start time and the current one before */
	  if((DBGetTimeInDouble(lEventNextTime) >= pmDrawStart)
	   &&(DBGetTimeInDouble(lEventTime) < pmDrawStart))
	    {
	    /* We found the first event */
	    lFirstEvent = lEvent;
	    break;
	    }

	  /* Go to the next event */
	  lEvent = lEventNext;
	  }
	}
      else
	{
	/* Start at the next event since the current event could be the draw start event */
	DBEventNext(pmEventGraph->EventDB, &lEvent);

	/* Keep going on backwards in the trace */
	do
	  {
	  /* Get the event time */
	  DBEventTime(pmEventGraph->EventDB, &lEvent, &lEventTime);
	  DBEventTime(pmEventGraph->EventDB, &lEventPrev, &lEventPrevTime);

	  /* Is the current event past the draw start time and the previous one before */
	  if((DBGetTimeInDouble(lEventTime) >= pmDrawStart)
	   &&(DBGetTimeInDouble(lEventPrevTime) < pmDrawStart))
	    {
	    /* We found the first event */
	    lFirstEvent = lEventPrev;
	    break;
	    }

	  /* Go to the previous event */
	  lEvent = lEventPrev;

	  /* Is this the first event drawable */
	  if(DBEventsEqual(lEvent, pmEventGraph->EventDB->FirstEventWithProc))
	    {
	    lFirstEvent = lEvent;
	    break;
	    }
	  }  while(DBEventPrev(pmEventGraph->EventDB, &lEventPrev) == TRUE);
	} /* End of else */
      } /* End of else */
    } /* End of else */

  /* Did we find anything (Both are initialized as zero, so test if lFirstEvent remained zero) */
  if(DBEventsEqual(lFirstEvent, lLastEvent))
    return FALSE;

  /* Loop around the rest of the events start from the first event  */
  lEventNext = lEvent = lFirstEvent;
  do
    {
    /* Get the event time */
    DBEventTime(pmEventGraph->EventDB, &lEvent, &lEventTime);
    DBEventTime(pmEventGraph->EventDB, &lEventNext, &lEventNextTime);

    /* Is the next event past the draw end time and the current one before it */
    if((DBGetTimeInDouble(lEventNextTime) > pmDrawEnd)
    && (DBGetTimeInDouble(lEventTime) <= pmDrawEnd))
      /* We found the last event */
      break;

    /* Go to the next event */
    lEvent = lEventNext;
    } while(DBEventNext(pmEventGraph->EventDB, &lEventNext) == TRUE);

  /* Set the last event drawn */
  lLastEvent = lEventNext;

  /* Are the last events and first events the same */
  if(DBEventsEqual(lFirstEvent, lLastEvent))
    return FALSE;

  /* Update draw start time */
  pmEventGraph->DrawStartTime = pmDrawStart;

  /* Set and update the limit events found */
  *pmFirstEvent = pmEventGraph->DrawStartEvent = lFirstEvent;
  *pmLastEvent  = pmEventGraph->DrawEndEvent   = lLastEvent;

  /* Tell the caller that the search was successfull */
  return TRUE;
}

/******************************************************************
 * Function :
 *    EGIEventInfo()
 * Description :
 *    This function draws a clipped icon and text information on a
 *    Pixmap.
 * Parameters :
 * Return values :
 * History :
 * Note :
 ******************************************************************/
void EGIEventInfo(eventGraph* pmEventGraph,
		  PixAndMask  EventIcon,
		  guint       x,
		  guint       y,
		  gchar*      EventText,
		  event*      pmEvent)
{
  EventIconArea* IconXY;

  /* Set Clipping information */
  gdk_gc_set_clip_mask   (pmEventGraph->BackgroundGC, EventIcon.Mask);
  gdk_gc_set_clip_origin (pmEventGraph->BackgroundGC,
			  x, y - EGRAPH_TEXT_TO_LINE_DIST - EGRAPH_ICONS_HEIGHT);

  /* Draw an icon that represents the passed event */
  gdk_draw_pixmap(pmEventGraph->PixMap,
		  pmEventGraph->BackgroundGC,
		  EventIcon.Pixmap,
		  0, 0,
		  x, y - EGRAPH_TEXT_TO_LINE_DIST - EGRAPH_ICONS_HEIGHT,
		  EGRAPH_ICONS_WIDTH, EGRAPH_ICONS_HEIGHT);

  /* Stop clipping */
  gdk_gc_set_clip_mask (pmEventGraph->BackgroundGC, NULL);

  /* Draw the text to be displayed */
  gdk_draw_string(pmEventGraph->PixMap,
		  pmEventGraph->TextFont,
		  pmEventGraph->TextGC,
		  x + EGRAPH_ICONS_WIDTH,
		  y - EGRAPH_TEXT_TO_LINE_DIST,
		  EventText);

  /* allocate a new event coordinate item */
  IconXY = (EventIconArea*) g_malloc(sizeof(EventIconArea));

  /* Memorize icon coordinates and event pointer */
  IconXY->x               = x;
  IconXY->y               = y;
  IconXY->AssociatedEvent = *pmEvent;

  /* Append the coordinates to the event list */
  pmEventGraph->EventIconPos = g_list_prepend(pmEventGraph->EventIconPos, (gpointer) IconXY);
}

/******************************************************************
 * Function :
 *    EGIDrawEvents()
 * Description :
 * Parameters :
 * Return values :
 * History :
 * Note :
 *    A scheduling change event can be a control event if the event
 *    immediatly following is an entry of some kind for a process
 *    other than 0.
 ******************************************************************/
void EGIDrawEvents(eventGraph* pmEventGraph,
		   gdouble pmDrawStart,  gdouble pmDrawEnd,
		   event*  pmFirstEvent, event*  pmLastEvent)
{
  db*               pEventDB;               /* The event database */
  gint              lEInfo;                 /* Miscallaneous event information */
  gint              lPID = -1;              /* PID of event owner */
  gint              lLastCtrlEventID;       /* The event ID of the last control event */
  guint             x1, y1, x2, y2;         /* Origin and destination points for drawing */
  gchar             lString[30];            /* String used to display information on graph */
  gboolean          lFirstCtrlEvent = TRUE; /* We are currently dealing with the first control% event */
  event             lEvent;                 /* Generic event pointer */
  event             lLastCtrlEvent;         /* Last event that generated a change of control */
  GdkGC*            pTempGC;                /* Temporary graphic context */
  system*           pSystem;                /* System pointer */
  process*          pEventProcess = NULL;   /* Process associated with event */
  process*          pLastCtrlEventProcess;  /* Process to which last control event belongs */
  struct timeval    lTime;                  /* Generic timevalue */
  struct timeval    lTimeTemp;              /* Temporary timevalue */
  struct timeval    lLastCtrlEventTime;     /* The time of occurence of the last control event */
  eventDescription  lEventDesc;             /* Generic event decription */

#if 0
  printf("\n");
  DBEventTime(pmEventGraph->EventDB, pmFirstEvent, &lTime);  
  printf("First event time : (%ld, %ld) \n", lTime.tv_sec, lTime.tv_usec);
  DBEventTime(pmEventGraph->EventDB, pmLastEvent, &lTime);  
  printf("Last  event time : (%ld, %ld) \n", lTime.tv_sec, lTime.tv_usec);
#endif

  /* Initialize event database */
  pEventDB = pmEventGraph->EventDB;

  /* Initialize system pointer */
  pSystem = pmEventGraph->System;

  /* Make sure the event icon coordinates list is empty */
  if(pmEventGraph->EventIconPos)
    {
    /* Free all allocated coordinate items */
    g_list_foreach(pmEventGraph->EventIconPos, (GFunc) g_free, NULL);

    /* Free the GList list items */
    g_list_free(pmEventGraph->EventIconPos);

    /* Reset list pointer */
    pmEventGraph->EventIconPos = NULL;
    }

  /* Go to the first significant event */
  lLastCtrlEvent = lEvent = *pmFirstEvent;

  /* Set last x and y */
  x2 = 0;
  y2 = EGIFindKernelHeight(pmEventGraph);

  /* Is this the first schedule-in */
  if(!DBEventsEqual(pEventDB->FirstEventWithProc, lLastCtrlEvent))
    /* Go back to the last control event */
    while((DBEventControl(pEventDB, &lLastCtrlEvent, pSystem) != TRUE)
	&&(DBEventPrev(pEventDB, &lLastCtrlEvent) == TRUE));

  /* Get the last control event's ID and it's process */
  lLastCtrlEventID = DBEventID(pEventDB, &lLastCtrlEvent);
  pLastCtrlEventProcess = DBEventProcess(pEventDB, &lLastCtrlEvent, pSystem);

  /* Get the current event's ID and it's process */
  pEventProcess = DBEventProcess(pEventDB, &lEvent, pSystem);
  lPID = pEventProcess->PID;

  /* Draw the events */
  while((DBEventNext(pEventDB, &lEvent) == TRUE)
     && (!DBEventsEqual(lEvent, (*pmLastEvent))))
    {
    /* Get the event's description */
    DBEventDescription(pmEventGraph->EventDB, &lEvent, FALSE, &lEventDesc);

    /* If the trace contains many CPUs only draw the first one */
    if((pmEventGraph->EventDB->LogCPUID == TRUE)
     &&(lEventDesc.CPUID != 0))
      continue;

#if 0
    printf("X1 = %u; X2 = %u; Y1 = %u; Y2 = %u \n", x1, x2, y1, y2);
#endif

    /* Depending on the event */
    switch(lEventDesc.ID)
      {
      /* Entry (syscall / trap / irq) */
      case TRACE_SYSCALL_ENTRY :
      case TRACE_TRAP_ENTRY :
      case TRACE_IRQ_ENTRY :
	/* Is this an IRQ entry while we were in the kernel */
	if((lEventDesc.ID == TRACE_IRQ_ENTRY)
	 &&(IRQ_EVENT(lEventDesc.Struct)->kernel))
	  {
	  /* Display the syscall entry information */
	  sprintf(lString, "%d", IRQ_EVENT(lEventDesc.Struct)->irq_id);

	  /* Draw the IRQ details */
	  EGIEventInfo(pmEventGraph,
		       pmEventGraph->IRQ,
		       EGIFindEventDistance(pmEventGraph, lEventDesc, pmDrawStart, pmDrawEnd),
		       y2 - EGRAPH_TEXT_TO_LINE_DIST,
		       lString,
		       &lEvent);

	  /* Don't go any further */
	  continue;
	  }

	/* Are we at the begining of the graph */
	if(lFirstCtrlEvent == TRUE)
	  {
	  /* Was the last control event an exit */
	  if(DBEventControlExit(pEventDB, &lLastCtrlEvent) == TRUE)
	    {
	    /* Find the height of the process */
	    y2 = EGIFindProcessHeight(pmEventGraph, pEventProcess);
	  
	    /* Use the process GC to draw the next horizontal line */
	    pTempGC = pmEventGraph->ProcessGC;
	    }
	  else
	    {
	    /* Find the height of the kernel */
	    y2 = EGIFindKernelHeight(pmEventGraph);
	  
	    /* Use the process GC to draw the next horizontal line */
	    pTempGC = pmEventGraph->KernelGC;
	    }
	  }
	else
	  {
	  /* Was the last control event an exit */
	  if(DBEventControlExit(pEventDB, &lLastCtrlEvent, lPID) == TRUE)
	    {
	    /* Draw a line from the irq exit or trap exit to the process */
	    x1 = x2;
	    y1 = y2;
	    x2 = x1;
	    y2 = EGIFindProcessHeight(pmEventGraph, pEventProcess);
	    switch(lLastCtrlEventID)
	      {
	      case TRACE_SYSCALL_EXIT :
		pTempGC = pmEventGraph->SysCallGC;
		break;
		
	      case TRACE_TRAP_EXIT :
		pTempGC = pmEventGraph->TrapGC;
		break;

	      default :
		pTempGC = pmEventGraph->InterruptGC;
		break;
	      }
	    gdk_draw_line(pmEventGraph->PixMap,
			  pTempGC,
			  x1,
			  y1,
			  x2,
			  y2);

	    /* Does this event belong to a process or to the kernel */
	    if((pEventProcess != NULL)
	     &&(pEventProcess->PID != 0))
	      /* Use the process GC to draw the next horizontal line */
	      pTempGC = pmEventGraph->ProcessGC;
	    else
	      /* Use the kernel GC to draw the next horizontal line */
	      pTempGC = pmEventGraph->KernelGC;
	    } /* end of else-if */
	  /* Was the last event an entry or something else */
	  else
	    /* Use the kernel GC to draw the next horizontal line */
	    pTempGC = pmEventGraph->KernelGC;
	  }
	
	/* Draw a line from the last exit to now */
	x1 = x2;
	y1 = y2;
	x2 = EGIFindEventDistance(pmEventGraph, lEventDesc, pmDrawStart, pmDrawEnd);
	y2 = y1;
	EGIDrawThickHLine(pmEventGraph->PixMap,
			  pTempGC,
			  x1,
			  y1,
			  x2,
			  y2);

	/* Display the event information, depending on it's type */
	switch(lEventDesc.ID)
	  {
	  case TRACE_SYSCALL_ENTRY :
	    /* Display the syscall entry information */
	    sprintf(lString, "%s", SyscallID[SYSCALL_EVENT(lEventDesc.Struct)->syscall_id]);

	    /* Draw the text and its icon */
	    EGIEventInfo(pmEventGraph,
			 pmEventGraph->SysCall,
			 x2,
			 y2,
			 lString,
			 &lEvent);
	    
	    /* Set temp GC */
	    pTempGC = pmEventGraph->SysCallGC;
	    break;

	  case TRACE_TRAP_ENTRY :
	    /* Display the syscall entry information */
	    sprintf(lString, "%s", TrapID[TRAP_EVENT(lEventDesc.Struct)->trap_id]);
	    
	    /* Draw the text and its icon */
	    EGIEventInfo(pmEventGraph,
			 pmEventGraph->Trap,
			 x2,
			 y2,
			 lString,
			 &lEvent);

	    /* Set temp GC */
	    pTempGC = pmEventGraph->TrapGC;
	    break;
	    
	  case TRACE_IRQ_ENTRY :
	    /* Display the syscall entry information */
	    sprintf(lString, "%d", IRQ_EVENT(lEventDesc.Struct)->irq_id);

	    /* If we're at kernel height, then back up a little */
	    if(y2 == EGIFindKernelHeight(pmEventGraph))
	      /* Draw the string */
	      EGIEventInfo(pmEventGraph,
			   pmEventGraph->IRQ,
			   x2,
			   y2 - EGRAPH_TEXT_TO_LINE_DIST,
			   lString,
			   &lEvent);
	    else
	      /* Draw the string */
	      EGIEventInfo(pmEventGraph,
			   pmEventGraph->IRQ,
			   x2,
			   y2,
			   lString,
			   &lEvent);
	    
	    /* Set current GC */
	    pTempGC = pmEventGraph->InterruptGC;
	    break;
	  }

	/* Draw a line from the process to the kernel */
	x1 = x2;
	y1 = y2;
	x2 = x1;
	y2 = EGIFindKernelHeight(pmEventGraph);
	gdk_draw_line(pmEventGraph->PixMap,
		      pTempGC,
		      x1,
		      y1,
		      x2,
		      y2);

	/* Is this a control event */
	if(DBEventControlEntry(pEventDB, &lEvent, lPID) == TRUE)
	  {
	  /* The first control event has been processed */
	  lFirstCtrlEvent = FALSE;

	  /* Set last control event */
	  lLastCtrlEvent = lEvent;

	  /* Get a pointer to the process to which this event belongs and the event's description */
	  pLastCtrlEventProcess = pEventProcess;
	  lLastCtrlEventID = lEventDesc.ID;
	  }
	break;

      /* Exit (syscall / trap /irq) */
      case TRACE_SYSCALL_EXIT :
      case TRACE_TRAP_EXIT :
      case TRACE_IRQ_EXIT :
	/* Draw a line from the last event to now */
	x1 = x2;
	y1 = y2;
	x2 = EGIFindEventDistance(pmEventGraph, lEventDesc, pmDrawStart, pmDrawEnd);
	y2 = y1;
	EGIDrawThickHLine(pmEventGraph->PixMap,
			  pmEventGraph->KernelGC,
			  x1,
			  y1,
			  x2,
			  y2);

	/* Is this a control event */
	if(DBEventControlExit(pEventDB, &lEvent, lPID) == TRUE)
	  {
	  /* The first control event has been processed */
	  lFirstCtrlEvent = FALSE;

	  /* Set last control event */
	  lLastCtrlEvent = lEvent;

	  /* Get a pointer to the process to which this event belongs and the event's description */
	  pLastCtrlEventProcess = pEventProcess;
	  lLastCtrlEventID = lEventDesc.ID;
	  }
	break;

      /* Scheduling change / Kernel timer / Bottom Half / Process / Network */
      case TRACE_SCHEDCHANGE :
      case TRACE_KERNEL_TIMER :
      case TRACE_BOTTOM_HALF :
      case TRACE_PROCESS :
      case TRACE_NETWORK :
	/* Display information according to event */
	switch(lEventDesc.ID)
	  {
	  /* Scheduling change */
	  case TRACE_SCHEDCHANGE :
	    /* Get the PID of the incoming process */
	    lPID = SCHED_EVENT(lEventDesc.Struct)->in;

	    /* Get a pointer to this process */
	    pEventProcess = DBGetProcByPID(lPID, pSystem);

	    /* Display the PID of the incoming process */
	    sprintf(lString, "%d", lPID);

	    /* Draw the text and its icon */
	    EGIEventInfo(pmEventGraph,
			 pmEventGraph->SchedChange, 
			 EGIFindEventDistance(pmEventGraph, lEventDesc, pmDrawStart, pmDrawEnd),
			 EGIFindKernelHeight(pmEventGraph),
			 lString, 
			 &lEvent);
	    break;
	    
	  /* Kernel timer */
	  case TRACE_KERNEL_TIMER :
	    /* Display the kernel timer information */
	    lString[0] = '\0';
	    
	    /* Draw the text and its icon */
	    EGIEventInfo(pmEventGraph, pmEventGraph->KernelTimer,
			 EGIFindEventDistance(pmEventGraph, lEventDesc, pmDrawStart, pmDrawEnd),
			 EGIFindKernelHeight(pmEventGraph),
			 lString,
			 &lEvent);
	    break;

	  /* Bottom Half */
	  case TRACE_BOTTOM_HALF :
	    /* Is this the kernel timer */
	    if(*BH_EVENT(lEventDesc.Struct) == 0)
	      break;
	    
	    /* Display the bottom half information */
	    sprintf(lString, "%d", TRAP_EVENT(lEventDesc.Struct)->trap_id);
	    
	    /* Draw the text and its icon */
	    EGIEventInfo(pmEventGraph, pmEventGraph->BottomHalf, 
			 EGIFindEventDistance(pmEventGraph, lEventDesc, pmDrawStart, pmDrawEnd),
			 EGIFindKernelHeight(pmEventGraph),
			 lString,
			 &lEvent);
	    break;

	  /* Process */
	  case TRACE_PROCESS :
	    break;

	  /* Network */
	  case TRACE_NETWORK :
	    break;
	  }	  

	/* Is this a control event resulting in a return to a process */
	if(DBEventControlExit(pEventDB, &lEvent, lPID) == TRUE)
	  {
	  /* Draw a line from the entry to the event */
	  x1 = x2;
	  y1 = y2;
	  x2 = EGIFindEventDistance(pmEventGraph, lEventDesc, pmDrawStart, pmDrawEnd);
	  y2 = y1;
	  EGIDrawThickHLine(pmEventGraph->PixMap,
			    pmEventGraph->KernelGC,
			    x1,
			    y1,
			    x2,
			    y2);
	  
	  /* Draw a line from the event change to the process */
	  x1 = x2;
	  y1 = y2;
	  x2 = x1;
	  y2 = EGIFindProcessHeight(pmEventGraph, pEventProcess);
	  gdk_draw_line(pmEventGraph->PixMap,
			pmEventGraph->InterruptGC,
			x1,
			y1,
			x2,
			y2);

	  /* The first control event has been processed */
	  lFirstCtrlEvent = FALSE;

	  /* Set last control event */
	  lLastCtrlEvent = lEvent;

	  /* Get a pointer to the process to which this event belongs and the event's description */
	  pLastCtrlEventProcess = pEventProcess;
	  lLastCtrlEventID = lEventDesc.ID;
	  }
	break;

      /* File system */
      case TRACE_FILE_SYSTEM :
	/* Are we starting to wait for I/O */
	if(FS_EVENT(lEventDesc.Struct)->event_sub_id == TRACE_FILE_SYSTEM_BUF_WAIT_START)
	  {
	  sprintf(lString, "I/O:START");
	  EGIEventInfo(pmEventGraph, pmEventGraph->IOStart,
		       EGIFindEventDistance(pmEventGraph, lEventDesc, pmDrawStart, pmDrawEnd),
		       EGIFindKernelHeight(pmEventGraph),
		       lString,
		       &lEvent);

	  }
	   
	/* Are we ending wait for I/O */
	if(FS_EVENT(lEventDesc.Struct)->event_sub_id == TRACE_FILE_SYSTEM_BUF_WAIT_END)
	  {
	  sprintf(lString, "I/O:END");
	  EGIEventInfo(pmEventGraph, pmEventGraph->IOEnd,
		       EGIFindEventDistance(pmEventGraph, lEventDesc, pmDrawStart, pmDrawEnd),
		       EGIFindKernelHeight(pmEventGraph),
		       lString,
		       &lEvent);
	  }
	break;

      /* Timer */
      case TRACE_TIMER :
	break;

      /* Memory */
      case TRACE_MEMORY :
	break;

      /* Socket */
      case TRACE_SOCKET :
	break;

      /* IPC */
      case TRACE_IPC :
	break;

      /* Otherwise */
      default :
	g_print("Encountered unknow event \n");
	exit(1);
	break;
      } /* End switch type of the event to draw */
    } /* End drawing loop */

  /* Did we draw all the event in the window */
  if(DBEventsEqual((*pmFirstEvent), pEventDB->FirstEvent)
   &&DBEventsEqual((*pmLastEvent),  pEventDB->LastEvent))
    /* Don't draw anything */
    return;

  /* Get the last control event's time */
  DBEventTime(pEventDB, &lLastCtrlEvent, &lLastCtrlEventTime);

#if 0
  printf("Last control event was %s \n", EventID[lLastCtrlEventID]);
#endif

  /* Was the last control event really a control event */
  if(DBEventControl(pEventDB, &lLastCtrlEvent, pSystem) == TRUE)
    {
    /* Was the last control event an exit */
    if(DBEventControlExit(pEventDB, &lLastCtrlEvent) == TRUE)
      {
      /* Choose the right GC to draw */
      switch(lLastCtrlEventID)
	{
	case TRACE_SYSCALL_EXIT :
	  pTempGC = pmEventGraph->SysCallGC;
	  break;

	case TRACE_TRAP_EXIT :
	  pTempGC = pmEventGraph->TrapGC;
	  break;

	default :
	  pTempGC = pmEventGraph->InterruptGC;
	  break;
	}
      
      /* Draw a vertical line to the process */
      x1 = x2;
      y1 = y2 - 1;
      x2 = x1;
      y2 = EGIFindProcessHeight(pmEventGraph, pLastCtrlEventProcess);
      /* Is the event visible */
      if(DBGetTimeInDouble(lLastCtrlEventTime) >= pmDrawStart)
	gdk_draw_line(pmEventGraph->PixMap,
		      pTempGC,
		      x1,
		      y1,
		      x2,
		      y2);
	  
      /* Draw a horizontal line to the end */
      x1 = x2;
      y1 = y2;
      x2 = pmEventGraph->HDrawSize;
      y2 = y1;
      EGIDrawThickHLine(pmEventGraph->PixMap,
			pmEventGraph->ProcessGC,
			x1,
			y1,
			x2,
			y2);
      }
    else  /* Last control event was an entry into the kernel */
      {
      /* Draw a horizontal line to the end */
      x1 = x2;
      y1 = y2;
      x2 = pmEventGraph->HDrawSize;
      y2 = y1;
      EGIDrawThickHLine(pmEventGraph->PixMap,
			pmEventGraph->KernelGC,
			x1,
			y1,
			x2,
			y2);
      }
    }
  else /* This wasn't a control event */
    {
    /* Draw a horizontal line through the screen */
    x1 = 0;
      
    /* Were we idle all this time or is this the first control event */
    if((pLastCtrlEventProcess != NULL)
     &&(pLastCtrlEventProcess != 0)
     &&(!DBEventsEqual(lLastCtrlEvent, pEventDB->FirstEventWithProc)))
      {
      /* This was a process */
      y1 = EGIFindProcessHeight(pmEventGraph, pLastCtrlEventProcess);
      pTempGC = pmEventGraph->ProcessGC;
      }
    else
      {
      /* This was in the kernel */
      y1 = EGIFindKernelHeight(pmEventGraph);
      pTempGC = pmEventGraph->KernelGC;
      }

    /* Draw t'ill the end */
    x2 = pmEventGraph->HDrawSize;
    y2 = y1;

    /* Draw the thick line */
    EGIDrawThickHLine(pmEventGraph->PixMap,
		      pTempGC,
		      x1,
		      y1,
		      x2,
		      y2);
    }
}

/******************************************************************
 * Function :
 * Description :
 * Parameters :
 * Return values :
 * History :
 * Note :
 ******************************************************************/
void EGIDrawHorizon(eventGraph* pmEventGraph)
{
  process*    pProc;     /* Generic process pointer */

  /* Go through all the processes in the process list */
  for(pProc = pmEventGraph->System->Processes;
      pProc != NULL;
      pProc = pProc->Next)
    {
    /* Is this process zero */
    if(pProc->PID != 0)
      gdk_draw_line(pmEventGraph->PixMap,
		    pmEventGraph->HorizonGC,
		    0,
		    EGIFindProcessHeight(pmEventGraph, pProc),
		    pmEventGraph->HDrawSize,
		    EGIFindProcessHeight(pmEventGraph, pProc));
    else
      gdk_draw_line(pmEventGraph->PixMap,
		    pmEventGraph->HorizonGC,
		    0,
		    EGIFindKernelHeight(pmEventGraph),
		    pmEventGraph->HDrawSize,
		    EGIFindKernelHeight(pmEventGraph));
    }
}

/******************************************************************
 * Function :
 * Description :
 * Parameters :
 * Return values :
 * History :
 * Note :
 ******************************************************************/
void EGIDrawSelectionLine(eventGraph* pmEventGraph)
{
  /* If a process is selected */
  if(pmEventGraph->SelectedProc != NULL)
      gdk_draw_line(pmEventGraph->PixMap,
		    pmEventGraph->SelectedGC,
		    0,
		    EGIFindProcessHeight(pmEventGraph, pmEventGraph->SelectedProc),
		    pmEventGraph->HDrawSize,
		    EGIFindProcessHeight(pmEventGraph, pmEventGraph->SelectedProc));
}

/******************************************************************
 * Function :
 * Description :
 * Parameters :
 * Return values :
 * History :
 * Note :
 ******************************************************************/
void EGIDrawTrace(eventGraph* pmEventGraph, gboolean pmForce)
{
  gint             lXi, lYi;            /* Origin point from which drawing should start */
  event            lFirstEvent;         /* First event drawn */
  event            lLastEvent;          /* Last event drawn */
  gchar            lSBarString[80];     /* String displayed in status bar */
  gfloat           lScrollDisplacement; /* Displacement between last and current scroll positions */
  gfloat           lValue;              /* Current position of scrollbar */
  gfloat           lPageSize;           /* Size of scroll page */
  gfloat           lMinValue;           /* Minimum value of scroll bar position */
  gfloat           lMaxValue;           /* Maximum value of scroll bar position */
  gdouble          lDrawStart;          /* Time of first event drawn */
  gdouble          lDrawEnd;            /* Time of last event drawn */

  /* Get scrollbar parameters */
  lValue        = GTK_ADJUSTMENT(pmEventGraph->HAdjustment)->value;
  lPageSize     = GTK_ADJUSTMENT(pmEventGraph->HAdjustment)->page_size;
  lMinValue     = GTK_ADJUSTMENT(pmEventGraph->HAdjustment)->lower;
  lMaxValue     = GTK_ADJUSTMENT(pmEventGraph->HAdjustment)->upper;

  /* Get displacement */
  lScrollDisplacement = GTK_ADJUSTMENT(pmEventGraph->HAdjustment)->value - pmEventGraph->LastScrollVal;

#if 0
  /* Print out the sizes */
  g_print("Value %f \n", lValue);
  g_print("Page size %f \n", lPageSize);
  g_print("Min Value %f \n", lMinValue);
  g_print("Max Value %f \n", lMaxValue);
#endif

  /* Is it worth it to redraw */
  if((lScrollDisplacement < 1) && (lScrollDisplacement > -1) && (pmForce != TRUE))
    return;

  /* Update the latest scroll value */
  pmEventGraph->LastScrollVal = lValue;

  /* Get the drawing interval */
  EGIFindDrawLimits(pmEventGraph, lValue, lPageSize, lMinValue, lMaxValue, &lDrawStart, &lDrawEnd);
  
  /* Print display interval in status bar */
  sprintf(lSBarString, "(%.0lf, %.0lf)", lDrawStart / 1000, lDrawEnd / 1000);
  WDStatusBarDisplay(pmEventGraph->MainWindow, lSBarString);

  /* Print out draw information */
#if 0
  g_print("Scroll Displacement \t: %f \n", lScrollDisplacement);
  g_print("Time Interval \t\t: %f \n",   pmEventGraph->Interval);
  g_print("Draw Start \t\t: %f \n",      lDrawStart);
  g_print("Draw End \t\t: %f \n",        lDrawEnd);
  g_print("Graph Start Time \t: %f \n",  pmEventGraph->StartTime);
  g_print("Graph End Time \t\t: %f \n",  pmEventGraph->StartTime + pmEventGraph->Duration);
  g_print("Graph Duration \t\t: %f \n",  pmEventGraph->Duration);
#endif

  if(pmEventGraph->Interval < 0)
    return;

  /* Find the first and last event drawable */
  if(EGIFindLimitEvents(pmEventGraph, lDrawStart, lDrawEnd, &lFirstEvent, &lLastEvent) == FALSE)
    {
    /* Draw all the events */
    lFirstEvent = pmEventGraph->EventDB->FirstEventWithProc;
    lLastEvent  = pmEventGraph->EventDB->LastEvent;
    }

  /* Empty the draw area */
  gdk_draw_rectangle(pmEventGraph->PixMap,
		     pmEventGraph->BackgroundGC,
		     TRUE,
		     0, 0,
		     pmEventGraph->HDrawSize,
		     pmEventGraph->VDrawSize);

#if 0
  g_print("First Event Drawn  : %f \n", EGIGetEventTimeInDouble(pFirstEvent));
  g_print("Last Event Drawn   : %f \n", EGIGetEventTimeInDouble(pLastEvent));
  g_print("NFirst Event Drawn : %f \n", EGIGetEventTimeInDouble(pFirstEvent->Next));
  g_print("PLast Event Drawn  : %f \n", EGIGetEventTimeInDouble(pLastEvent->Prev));
  g_print("Interval last/first : %f \n", EGIGetEventTimeInDouble(pLastEvent) - EGIGetEventTimeInDouble(pFirstEvent));
  g_print("Interval pre-set    : %f \n", pmEventGraph->Interval);
#endif

  /* Draw the horizon */
  if(pmEventGraph->ShowHorizon == TRUE)
    EGIDrawHorizon(pmEventGraph);

  /* Draw the viewable events */
  EGIDrawEvents(pmEventGraph,
		lDrawStart,  lDrawEnd,
		&lFirstEvent, &lLastEvent);

  /* Draw the selection line */
  EGIDrawSelectionLine(pmEventGraph);
}

/******************************************************************
 * Function :
 * Description :
 * Parameters :
 * Return values :
 * History :
 * Note :
 ******************************************************************/
void EGIResetHAdjustment(eventGraph* pmEGraph, gboolean pmKeepInRegion)
{
  /* Are we somewhere else than in the begining */
  if(pmEGraph->LastScrollVal != 0 && pmKeepInRegion)
    /* Keep the scrollbar in the same region */
    pmEGraph->LastScrollVal = 
      ((gfloat) pmEGraph->HTrueSize) *
      GTK_ADJUSTMENT(pmEGraph->HAdjustment)->value / 
      GTK_ADJUSTMENT(pmEGraph->HAdjustment)->upper;

#if 0
  g_print("\nBefore \n");
  g_print("Value : %f \n", GTK_ADJUSTMENT(pmEGraph->HAdjustment)->value);
  g_print("Page size %f \n", GTK_ADJUSTMENT(pmEGraph->HAdjustment)->page_size);
  g_print("Min Value %f \n", GTK_ADJUSTMENT(pmEGraph->HAdjustment)->lower);
  g_print("Max Value %f \n", GTK_ADJUSTMENT(pmEGraph->HAdjustment)->upper);
#endif

  /* Set scrollbar parameters */
  GTK_ADJUSTMENT(pmEGraph->HAdjustment)->lower = 0;
  GTK_ADJUSTMENT(pmEGraph->HAdjustment)->upper = (gfloat) pmEGraph->HTrueSize;
  GTK_ADJUSTMENT(pmEGraph->HAdjustment)->value =  pmEGraph->LastScrollVal;
  GTK_ADJUSTMENT(pmEGraph->HAdjustment)->page_size = (gfloat) pmEGraph->HDrawSize;
  GTK_ADJUSTMENT(pmEGraph->HAdjustment)->step_increment = 10;
  GTK_ADJUSTMENT(pmEGraph->HAdjustment)->page_increment = (gfloat) pmEGraph->HDrawSize; 
  gtk_adjustment_changed(GTK_ADJUSTMENT(pmEGraph->HAdjustment));

  /* Set drawable interval */
  pmEGraph->Interval = (pmEGraph->Duration * (gdouble) pmEGraph->HDrawSize) / ((gdouble) pmEGraph->HTrueSize);

#if 0
  g_print("After \n");
  g_print("Value : %f \n", GTK_ADJUSTMENT(pmEGraph->HAdjustment)->value);
  g_print("Page size %f \n", GTK_ADJUSTMENT(pmEGraph->HAdjustment)->page_size);
  g_print("Min Value %f \n", GTK_ADJUSTMENT(pmEGraph->HAdjustment)->lower);
  g_print("Max Value %f \n", GTK_ADJUSTMENT(pmEGraph->HAdjustment)->upper);
#endif
}

/**********************************************************************************/
/**************************** Signal handling functions ***************************/
/**********************************************************************************/
/******************************************************************
 * Function :
 * Description :
 * Parameters :
 * Return values :
 * History :
 * Note :
 ******************************************************************/
void EGSHExposeEvent(GtkWidget* pmWidget, GdkEventExpose* pmEvent, eventGraph* pmEGraph)
{
  gfloat         lRStart;     /* Start value of ruler */
  gfloat         lREnd;       /* End value of ruler */

#if 0
  g_print("Expose event \n");
#endif

  /* Are we ready to expose something */
  if((pmEGraph->System == NULL)
  || (pmEGraph->PixMap == NULL)
  || (pmEGraph->Init   == FALSE))
    return;
  
  /* Draw the pixmap in the window */
  gdk_draw_pixmap(pmEGraph->DrawArea->window,
		  pmEGraph->DrawArea->style->fg_gc[GTK_WIDGET_STATE(pmEGraph->DrawArea)],
		  pmEGraph->PixMap,
#if 0
		  pmEvent->area.x, pmEvent->area.y,
		  pmEvent->area.x, pmEvent->area.y,
		  pmEvent->area.width, pmEvent->area.height);
#else
                  0, 0,
                  0, 0,
		  pmEGraph->HDrawSize, pmEGraph->VDrawSize);
#endif

  /* Set the ruler's values */
  if(pmEGraph->DrawStartTime != 0)
    {
    /* Are we displaying in seconds */
    if(pmEGraph->Interval >= 1000000)
      {
      lRStart = (gfloat) (pmEGraph->DrawStartTime / 1000000) / 1000;
      lREnd = lRStart + (gfloat) (pmEGraph->Interval / 1000000) / 1000;
      }
    else  /* Microseconds */
      {
      lRStart = (gfloat) fmod(pmEGraph->DrawStartTime, 1000000) / 1000;
      lREnd = (gfloat) fmod(pmEGraph->DrawStartTime + pmEGraph->Interval, 1000000) / 1000;
      }
    if(lREnd < lRStart)
      lREnd += 1000;
    gtk_ruler_set_range(GTK_RULER(pmEGraph->HRuler), lRStart, lREnd, lRStart, lREnd);
    }

  /* Make sure that the scrolled process list is well drawn (if not done, list isn't properly sized !?!?) */
  gtk_widget_queue_resize(pmEGraph->ScrolledList);

  /* Stop the signal from propagating */
  gtk_signal_emit_stop_by_name(GTK_OBJECT(pmWidget), "expose_event");
}

/******************************************************************
 * Function :
 * Description :
 * Parameters :
 * Return values :
 * History :
 * Note :
 ******************************************************************/
void EGSHConfigureEvent(GtkWidget* pmWidget, GdkEventExpose* pmEvent, eventGraph* pmEGraph)
{
  guint    lOldHDrawSize;     /* Old horizontal draw size */
  guint    lOldVDrawSize;     /* Old vertical draw size */

#if 0
  g_print("Configure event \n");
#endif

  /* Are we ready to expose something */
  if((pmEGraph->System == NULL)
  || (pmEGraph->Init == FALSE))
    return;

  /* Remember old draw sizes */
  lOldHDrawSize = pmEGraph->HDrawSize;
  lOldVDrawSize = pmEGraph->VDrawSize;

  /* Set new draw sizes */
  pmEGraph->HDrawSize = (guint) GTK_WIDGET(pmEGraph->DrawArea)->allocation.width;
  pmEGraph->VDrawSize = (guint) GTK_WIDGET(pmEGraph->DrawArea)->allocation.height;

  /* Do we need more space than is available */
  if(pmEGraph->VDrawSize < pmEGraph->VTrueSize)
     pmEGraph->VDrawSize = pmEGraph->VTrueSize;

  /* Does the horizontal drawing area have a significant size */
#if 0
  if(pmEGraph->HDrawSize < 10)
     return;
#endif

  /* Have the draw sizes changed */
  if((lOldHDrawSize == pmEGraph->HDrawSize)
  && (lOldVDrawSize == pmEGraph->VDrawSize))
    {
    /* Reset the HAdjustment */
    EGIResetHAdjustment(pmEGraph, TRUE);

    /* Get outta here */
    return;
    }

  /* Was there a pixmap already */
  if(pmEGraph->PixMap != NULL)
    {
    /* Free the already existing pixmap */
    gdk_pixmap_unref(pmEGraph->PixMap);
    pmEGraph->PixMap = NULL;
    }

  /* Create new pixmap where we can draw */
  pmEGraph->PixMap = gdk_pixmap_new(pmEGraph->DrawArea->window,
				    pmEGraph->HDrawSize,
				    pmEGraph->VDrawSize,
				    -1);

  /* Set draw area size  (if this isn't done, scrolling down doesn't enable the user to see more of the graph) */
  gtk_widget_set_usize(pmEGraph->DrawArea,
		       -1,
		       pmEGraph->VDrawSize);

  /* Did we have cached icons? */
  if(pmEGraph->KernelTimer.Pixmap != NULL)
    {
    /* Free the already existing icons */
    gdk_pixmap_unref(pmEGraph->KernelTimer.Pixmap);
    gdk_pixmap_unref(pmEGraph->SysCall.Pixmap);
    gdk_pixmap_unref(pmEGraph->Trap.Pixmap);
    gdk_pixmap_unref(pmEGraph->SchedChange.Pixmap);
    gdk_pixmap_unref(pmEGraph->BottomHalf.Pixmap);
    gdk_pixmap_unref(pmEGraph->IOStart.Pixmap);
    gdk_pixmap_unref(pmEGraph->IOEnd.Pixmap);
    gdk_pixmap_unref(pmEGraph->IRQ.Pixmap);

    gdk_bitmap_unref(pmEGraph->KernelTimer.Mask);
    gdk_bitmap_unref(pmEGraph->SysCall.Mask);
    gdk_bitmap_unref(pmEGraph->Trap.Mask);
    gdk_bitmap_unref(pmEGraph->SchedChange.Mask);
    gdk_bitmap_unref(pmEGraph->BottomHalf.Mask);
    gdk_bitmap_unref(pmEGraph->IOStart.Mask);
    gdk_bitmap_unref(pmEGraph->IOEnd.Mask);
    gdk_bitmap_unref(pmEGraph->IRQ.Mask);

    pmEGraph->KernelTimer.Pixmap = NULL;
    pmEGraph->SysCall.Pixmap     = NULL;
    pmEGraph->Trap.Pixmap        = NULL;
    pmEGraph->SchedChange.Pixmap = NULL;
    pmEGraph->BottomHalf.Pixmap  = NULL;
    pmEGraph->IOStart.Pixmap     = NULL;
    pmEGraph->IOEnd.Pixmap       = NULL;
    pmEGraph->IRQ.Pixmap         = NULL;
    }
    
  /* Create cached icons */
  pmEGraph->KernelTimer.Pixmap = gdk_pixmap_create_from_xpm_d( pmEGraph->DrawArea->window,
                                                              &pmEGraph->KernelTimer.Mask,
                                                              NULL,
                                                              xpm_kernel_timer );
  pmEGraph->SysCall.Pixmap     = gdk_pixmap_create_from_xpm_d( pmEGraph->DrawArea->window,
                                                              &pmEGraph->SysCall.Mask,
                                                              NULL,
                                                              xpm_sys_call );
  pmEGraph->Trap.Pixmap        = gdk_pixmap_create_from_xpm_d( pmEGraph->DrawArea->window,
                                                              &pmEGraph->Trap.Mask,
                                                              NULL,
                                                              xpm_trap );
  pmEGraph->SchedChange.Pixmap = gdk_pixmap_create_from_xpm_d( pmEGraph->DrawArea->window,
                                                              &pmEGraph->SchedChange.Mask,
                                                              NULL,
                                                              xpm_sched_change );
  pmEGraph->BottomHalf.Pixmap  = gdk_pixmap_create_from_xpm_d( pmEGraph->DrawArea->window,
                                                              &pmEGraph->BottomHalf.Mask,
                                                              NULL,
                                                              xpm_bottom_half );
  pmEGraph->IOStart.Pixmap     = gdk_pixmap_create_from_xpm_d( pmEGraph->DrawArea->window,
                                                              &pmEGraph->IOStart.Mask,
                                                              NULL,
                                                              xpm_io_start );
  pmEGraph->IOEnd.Pixmap       = gdk_pixmap_create_from_xpm_d( pmEGraph->DrawArea->window,
                                                              &pmEGraph->IOEnd.Mask,
                                                              NULL,
                                                              xpm_io_end );
  pmEGraph->IRQ.Pixmap         = gdk_pixmap_create_from_xpm_d( pmEGraph->DrawArea->window,
                                                              &pmEGraph->IRQ.Mask,
                                                              NULL,
                                                              xpm_irq );

  /* Have the colors been allocated */
  if(pmEGraph->ColorsAllocated == FALSE)
    {
    /* Create the Graphic Context for the Selected item */

    /* Create default drawing contexts */
    pmEGraph->BackgroundGC= EGICreateGC(pmEGraph->PixMap,
					pmEGraph->BlackColor,
					pmEGraph->BlackColor, 
					GDK_LINE_SOLID);
    pmEGraph->HorizonGC   = EGICreateGC(pmEGraph->PixMap,
					pmEGraph->YellowColor,
					pmEGraph->BlackColor, 
					GDK_LINE_SOLID);
    pmEGraph->ProcessGC   = EGICreateGC(pmEGraph->PixMap,
					pmEGraph->GreenColor,
					pmEGraph->BlackColor,
					GDK_LINE_SOLID);
    pmEGraph->Process0GC  = EGICreateGC(pmEGraph->PixMap,
					pmEGraph->RedColor,
					pmEGraph->BlackColor,
					GDK_LINE_SOLID);
    pmEGraph->KernelGC    = EGICreateGC(pmEGraph->PixMap,
					pmEGraph->OrangeColor,
					pmEGraph->BlackColor,
					GDK_LINE_SOLID);
    pmEGraph->SysCallGC   = EGICreateGC(pmEGraph->PixMap,
					pmEGraph->BlueColor,
					pmEGraph->BlackColor,
					GDK_LINE_SOLID);
    pmEGraph->TrapGC      = EGICreateGC(pmEGraph->PixMap,
					pmEGraph->GreyColor,
					pmEGraph->BlackColor,
					GDK_LINE_SOLID);
    pmEGraph->InterruptGC = EGICreateGC(pmEGraph->PixMap,
					pmEGraph->WhiteColor,
					pmEGraph->BlackColor,
					GDK_LINE_SOLID);
    pmEGraph->TextGC      = EGICreateGC(pmEGraph->PixMap,
					pmEGraph->TurquoiseColor,
					pmEGraph->BlackColor,
					GDK_LINE_SOLID);
    pmEGraph->SelectedGC  = EGICreateGC(pmEGraph->PixMap,
					pmEGraph->PurpleColor,
					pmEGraph->BlackColor,
					GDK_LINE_ON_OFF_DASH);

    /* We're done allocating colors */
    pmEGraph->ColorsAllocated = TRUE;
    }

  /* Reset HAdjustment settings */
  EGIResetHAdjustment(pmEGraph, TRUE);

  /* Draw the trace */
  EGIDrawTrace(pmEGraph, TRUE);

  /* Stop the signal from propagating */
  gtk_signal_emit_stop_by_name(GTK_OBJECT(pmWidget), "configure_event");
}

/******************************************************************
 * Function :
 * Description :
 * Parameters :
 * Return values :
 * History :
 * Note :
 ******************************************************************/
void EGSHHScrollChange(GtkAdjustment* pmAdjustment, gpointer pmData)
{
  eventGraph*    pEGraph;     /* The event graph */
  GtkAdjustment* pHAdj;       /* An adjustment */

  /* Get the event graph */
  if((pEGraph = (eventGraph*) pmData) == NULL)
    return;

  /* Are we ready to expose something */
  if((pEGraph->System == NULL)
  || (pEGraph->Init == FALSE))
    return;

  /* Redraw trace */
  EGIDrawTrace(pEGraph, FALSE);

  /* Draw the pixmap in the window */
  gtk_widget_queue_draw(pEGraph->DrawArea);

  /* Stop the signal from being propagated */
  gtk_signal_emit_stop_by_name(GTK_OBJECT(pmAdjustment), "value-changed");
}

/******************************************************************
 * Function :
 * Description :
 * Parameters :
 * Return values :
 * History :
 * Note :
 ******************************************************************/
void EGSHSelectListItem(GtkWidget* pmListItem, gpointer pmData)
{
  eventGraph*    pEGraph;     /* The event graph */

  /* Get the event graph */
  if((pEGraph = (eventGraph*) pmData) == NULL)
    return;

  /* Get the selected process */
  pEGraph->SelectedProc = (process *) gtk_object_get_data(GTK_OBJECT(pmListItem),
							  gProcessItemData);

  /* Redraw the event graph */
  EGIDrawTrace(pEGraph, TRUE);

  /* Make sure that the changes are seen on the draw area */
  gtk_widget_queue_resize(pEGraph->DrawArea);
}

/**********************************************************************************/
/******************************* Windowing functions ******************************/
/**********************************************************************************/
/******************************************************************
 * Function :
 *     EGMouseOnIcon()
 * Description :
 *     Tests for the presence of the mouse pointer within an eventgraph
 *     icon
 * Parameters :
 *     x, y:   Coordinates of the mouse pointer
 *     coords: EventIconArea of the icon to test for
 * Return values :
 *     TRUE if within coordinates, FALSE otherwise
 * History :
 * Note :
 ******************************************************************/
gboolean EGMouseOnIcon(gint x, gint y, EventIconArea* coords)
{ 
  return ((x >= coords->x && x <= (coords->x + EGRAPH_ICONS_WIDTH))
	&&(y <= coords->y && y >= (coords->y - EGRAPH_ICONS_HEIGHT)));
}

/******************************************************************
 * Function :
 * Description :
 * Parameters :
 * Return values :
 * History :
 * Note :
 ******************************************************************/
void EGZoomIn(eventGraph* pmEGraph)
{
  /* Change the number of micro-seconds per pixel */
  pmEGraph->USecPerPix = pmEGraph->USecPerPix / EGRAPH_ZOOM_IN_FACTOR;

  /* Is the new number of usec/pix too small */
  if(pmEGraph->USecPerPix < EGRAPH_USEC_PER_PIX_MIN)
    pmEGraph->USecPerPix = EGRAPH_USEC_PER_PIX_MIN;

  /* Set the new true size */
  pmEGraph->HTrueSize = (gulong) (pmEGraph->Duration / (gdouble) pmEGraph->USecPerPix);

  /* Reset the horizontal scrollbar */
  EGIResetHAdjustment(pmEGraph, TRUE);

  /* Redraw the trace */
  EGIDrawTrace(pmEGraph, TRUE);

  /* Update the draw area per se */
  gtk_widget_queue_resize(pmEGraph->DrawArea);
}

/******************************************************************
 * Function :
 * Description :
 * Parameters :
 * Return values :
 * History :
 * Note :
 ******************************************************************/
void EGZoomOut(eventGraph* pmEGraph)
{
  /* Change the number of micro-seconds per pixel */
  pmEGraph->USecPerPix = pmEGraph->USecPerPix * EGRAPH_ZOOM_OUT_FACTOR;

  /* Is the new number of usec/pix too small */
  if(pmEGraph->USecPerPix > EGRAPH_USEC_PER_PIX_MAX)
    pmEGraph->USecPerPix = EGRAPH_USEC_PER_PIX_MAX;

  /* Set the new true size */
  pmEGraph->HTrueSize = (gulong) (pmEGraph->Duration / (gdouble) pmEGraph->USecPerPix);

  /* Reset the horizontal scrollbar */
  EGIResetHAdjustment(pmEGraph, TRUE);

  /* Redraw the trace */
  EGIDrawTrace(pmEGraph, TRUE);

  /* Update the draw area per se */
  gtk_widget_queue_resize(pmEGraph->DrawArea);
}

/******************************************************************
 * Function :
 * Description :
 * Parameters :
 * Return values :
 * History :
 * Note :
 ******************************************************************/
void EGZoomTimeFrame(eventGraph* pmEGraph, gdouble StartTime, gdouble EndTime)
{
  gdouble Duration; /* The new value for the trace duration */
  gdouble TempSwap; /* Temporary value to swap variables */

  /* Restrict new times to allowed values */
  if(StartTime > EndTime)
    {
    TempSwap  = EndTime;
    EndTime   = StartTime;
    StartTime = TempSwap;
    }
  if(StartTime < pmEGraph->StartTime || StartTime > pmEGraph->StartTime + pmEGraph->Duration) 
    StartTime = pmEGraph->StartTime;
  if(EndTime < pmEGraph->StartTime || EndTime > pmEGraph->StartTime + pmEGraph->Duration)
    EndTime = pmEGraph->StartTime + pmEGraph->Duration;

  /* Calculate the newly desired duration */
  Duration = EndTime - StartTime;

  /* Change the number of micro-seconds per pixel */
  pmEGraph->USecPerPix = Duration / GTK_WIDGET(pmEGraph->DrawArea)->allocation.width;

  /* Is the new number of usec/pix too big */
  if(pmEGraph->USecPerPix > EGRAPH_USEC_PER_PIX_MAX)
    pmEGraph->USecPerPix = EGRAPH_USEC_PER_PIX_MAX;

  /* Is the new number of usec/pix too small */
  if(pmEGraph->USecPerPix < EGRAPH_USEC_PER_PIX_MIN)
    pmEGraph->USecPerPix = EGRAPH_USEC_PER_PIX_MIN;

  /* Set the new true size */
  pmEGraph->HTrueSize = (gulong) (pmEGraph->Duration / (gdouble) pmEGraph->USecPerPix);

  /* Set the new scroll value */
  pmEGraph->LastScrollVal = 
    ( ( (StartTime - pmEGraph->StartTime) / pmEGraph->Duration) * pmEGraph->HTrueSize);

  /* Reset the horizontal scrollbar */
  EGIResetHAdjustment(pmEGraph, FALSE);

  /* Redraw the trace */
  EGIDrawTrace(pmEGraph, TRUE);

  /* Update the draw area per se */
  gtk_widget_queue_resize(pmEGraph->DrawArea);
}

/******************************************************************
 * Function :
 * Description :
 * Parameters :
 * Return values :
 * History :
 * Note :
 ******************************************************************/
void EGScrollToEvent(eventGraph* pmEGraph, event* pmEvent)
{
  gfloat            NewValue;     /* The new value for the adjustment */
  gdouble           EventTime;    /* The time of the event */
  struct timeval    lTime;        /* Event time */

  /* Is the Event graph passed valid? */
  if(pmEGraph == NULL)
    {
    g_print("ScrollToEvent called without graph passed! \n");
    exit(1);
    }

  /* Get event time */
  DBEventTime(pmEGraph->EventDB, pmEvent, &lTime);

  /* Get the time to scroll to */
  EventTime = DBGetTimeInDouble(lTime);

  /* Place the adjustment at the desired position */
  NewValue = ( ( (EventTime - pmEGraph->StartTime) / pmEGraph->Duration) * 
               (GTK_ADJUSTMENT(pmEGraph->HAdjustment)->upper - GTK_ADJUSTMENT(pmEGraph->HAdjustment)->lower)
             ) + GTK_ADJUSTMENT(pmEGraph->HAdjustment)->lower
               - (GTK_ADJUSTMENT(pmEGraph->HAdjustment)->page_size / 2);

  /* confine the adjustment to reasonable values */
  if(NewValue < GTK_ADJUSTMENT(pmEGraph->HAdjustment)->lower)
    NewValue = GTK_ADJUSTMENT(pmEGraph->HAdjustment)->lower;
  if(NewValue > GTK_ADJUSTMENT(pmEGraph->HAdjustment)->upper)
    NewValue = GTK_ADJUSTMENT(pmEGraph->HAdjustment)->upper;

  /* Set the new adjustment */
  gtk_adjustment_set_value(GTK_ADJUSTMENT(pmEGraph->HAdjustment), NewValue);
}

/******************************************************************
 * Function :
 * Description :
 * Parameters :
 * Return values :
 * History :
 * Note :
 ******************************************************************/
void EGISetHorizonView(eventGraph* pmEventGraph)
{
  /* Is the button active or not (I know this sounds like silly code, but it's "safer" this way) */
  if(pmEventGraph->ShowHorizon == FALSE)
    pmEventGraph->ShowHorizon = FALSE;
  else
    pmEventGraph->ShowHorizon = TRUE;

  /* Redraw the event graph */
  EGIDrawTrace(pmEventGraph, TRUE);

  /* Make sure that the changes are seen on the draw area */
  gtk_widget_queue_resize(pmEventGraph->DrawArea);
}

/******************************************************************
 * Function :
 * Description :
 * Parameters :
 * Return values :
 * History :
 * Note :
 ******************************************************************/
void EGConnectSignals(eventGraph* pmEventGraph)
{
  /* Connect to the window events */
  gtk_signal_connect(GTK_OBJECT(pmEventGraph->DrawArea),
		     "expose_event",
		     GTK_SIGNAL_FUNC(EGSHExposeEvent),
		     pmEventGraph);
  gtk_signal_connect(GTK_OBJECT(pmEventGraph->DrawArea),
		     "configure_event",
		     GTK_SIGNAL_FUNC(EGSHConfigureEvent),
		     pmEventGraph);

  /* Enable draw area to cause various events to take place */
  gtk_widget_set_events(pmEventGraph->DrawArea, GDK_POINTER_MOTION_MASK    /* motion_notify_event */
			                      | GDK_BUTTON_PRESS_MASK);    /* button-press-event */

  /* Connect horizontal ruler */
  gtk_signal_connect_object(GTK_OBJECT(pmEventGraph->DrawArea),
			    "motion_notify_event",
			    GTK_SIGNAL_FUNC(GTK_WIDGET_CLASS(GTK_OBJECT(pmEventGraph->HRuler)->klass)->motion_notify_event),
			    GTK_OBJECT(pmEventGraph->HRuler));

  /* Connect to the horizontal adjustment */
  gtk_signal_connect(GTK_OBJECT(pmEventGraph->HAdjustment),
		     "value-changed",
		     GTK_SIGNAL_FUNC(EGSHHScrollChange),
		     pmEventGraph);
}

/******************************************************************
 * Function :
 * Description :
 * Parameters :
 * Return values :
 * History :
 * Note :
 ******************************************************************/
void EGShowEventGraph(eventGraph* pmEventGraph)
{
  /* Is there anything to be shown */
  if(!pmEventGraph)
    {
    g_print("pmEventGraph is null in EGShowEventGraph! \n");
    exit(1);
    };

  /* Show it all */
  gtk_widget_show(pmEventGraph->HPanned);
  gtk_widget_show(pmEventGraph->ProcVBox);
  gtk_widget_show(pmEventGraph->DummyBox);
  gtk_widget_show(pmEventGraph->DrawVBox);
  gtk_widget_show(pmEventGraph->ScrolledList);
  gtk_widget_show(pmEventGraph->List);
  gtk_widget_show(pmEventGraph->HRuler);
  gtk_widget_show(pmEventGraph->ScrolledDraw);
  gtk_widget_show(pmEventGraph->DrawArea);
  gtk_widget_show(pmEventGraph->HScrollBar);
}

/******************************************************************
 * Function :
 * Description :
 * Parameters :
 * Return values :
 * History :
 * Note :
 ******************************************************************/
eventGraph* EGCreateEventGraph(gpointer pmMainWindow)
{
  eventGraph*      pEGraph;       /* The event graph */
  mainWindow*      pMainWindow;   /* Main window to which event graph belongs */
  GtkRequisition   lRequisition;  /* Used to obtain widget sizes*/

  /* Allocate space for new even graph */
  pEGraph = (eventGraph*) g_malloc(sizeof(eventGraph));

  /* Get main window */
  pMainWindow = (mainWindow*) pmMainWindow;

  /* Set state */
  pEGraph->Init            = FALSE;
  pEGraph->ColorsAllocated = FALSE;
  pEGraph->ShowHorizon     = FALSE;

  /* Set data */
  pEGraph->EventDB      = NULL;
  pEGraph->System       = NULL;
  pEGraph->MainWindow   = pmMainWindow;
  pEGraph->EventIconPos = NULL;
  pEGraph->SelectedProc = NULL;

  /* Initialize icon cache */
  pEGraph->KernelTimer.Pixmap = NULL;
  pEGraph->SysCall.Pixmap     = NULL;
  pEGraph->Trap.Pixmap        = NULL;
  pEGraph->SchedChange.Pixmap = NULL;
  pEGraph->BottomHalf.Pixmap  = NULL;
  pEGraph->IOStart.Pixmap     = NULL;
  pEGraph->IOEnd.Pixmap       = NULL;
  pEGraph->IRQ.Pixmap         = NULL;

  /* Allocate main elements */
  pEGraph->HPanned = gtk_hpaned_new();
  gtk_box_pack_start(GTK_BOX(pMainWindow->GTHBox), pEGraph->HPanned, TRUE, TRUE, 0);

  /* Create boxes and put them in hpanned */
  pEGraph->ProcVBox = gtk_vbox_new(FALSE, 0);
  pEGraph->DummyBox = gtk_vbox_new(FALSE, 0);
  pEGraph->DrawVBox = gtk_vbox_new(FALSE, 0);
  gtk_paned_add1(GTK_PANED(pEGraph->HPanned), pEGraph->ProcVBox);
  gtk_paned_add2(GTK_PANED(pEGraph->HPanned), pEGraph->DrawVBox);

  /* Create scrolled list and place it in the hpanned widget */
  pEGraph->ScrolledList = gtk_scrolled_window_new(NULL, NULL);
  pEGraph->List         = gtk_list_new();
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(pEGraph->ScrolledList),
				 GTK_POLICY_ALWAYS,
				 GTK_POLICY_ALWAYS);
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(pEGraph->ScrolledList),
					pEGraph->List);
  gtk_widget_set_usize(GTK_WIDGET(pEGraph->ScrolledList), 150, 0);

  /* Create horizontal ruler */
  pEGraph->HRuler = gtk_hruler_new();

  /* Set dummy box size */
  gtk_widget_size_request(GTK_WIDGET(pEGraph->HRuler), &lRequisition);
  gtk_widget_set_usize(GTK_WIDGET(pEGraph->DummyBox), 0, lRequisition.height);
  /*  gtk_box_set_spacing(GTK_BOX(pEGraph->ProcVBox), lRequisition.height);*/ /* This can work too */

  /* Put scrolled list in VBox window */
  gtk_box_pack_start(GTK_BOX(pEGraph->ProcVBox), pEGraph->DummyBox, FALSE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(pEGraph->ProcVBox), pEGraph->ScrolledList, TRUE, TRUE, 0);

  /* Create drawing area elements and place them in the hpanned widget */
  pEGraph->DrawArea     = gtk_drawing_area_new();
  pEGraph->ScrolledDraw = gtk_scrolled_window_new
                          (NULL,
			   gtk_scrolled_window_get_vadjustment(GTK_SCROLLED_WINDOW(pEGraph->ScrolledList)));
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(pEGraph->ScrolledDraw),
				 GTK_POLICY_NEVER,
				 GTK_POLICY_NEVER);
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(pEGraph->ScrolledDraw),
					pEGraph->DrawArea);

  /* Create horizontal scrollbar */
  pEGraph->HAdjustment = gtk_adjustment_new(0,0,0,0,0,0);
  pEGraph->HScrollBar  = gtk_hscrollbar_new(GTK_ADJUSTMENT(pEGraph->HAdjustment));

  /* Put ruler and draw area in hbox */
  gtk_box_pack_start(GTK_BOX(pEGraph->DrawVBox), pEGraph->HRuler, FALSE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(pEGraph->DrawVBox), pEGraph->ScrolledDraw, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(pEGraph->DrawVBox), pEGraph->HScrollBar, FALSE, TRUE, 0);

  /* Set pixmap */
  pEGraph->PixMap = NULL;

  /* Create the colors */
  pEGraph->RedColor       = EGICreateColor(0xffff, 0, 0);  /* (red, green, blue) */
  pEGraph->GreenColor     = EGICreateColor(0, 0xffff, 0);
  pEGraph->BlueColor      = EGICreateColor(0, 0, 0xffff);
  pEGraph->GreyColor      = EGICreateColor(0x8000, 0x8000, 0x8000);
  pEGraph->GreenColor     = EGICreateColor(0, 0xffff, 0);
  pEGraph->BlackColor     = EGICreateColor(0, 0, 0);
  pEGraph->WhiteColor     = EGICreateColor(0xffff, 0xffff, 0xffff);
  pEGraph->OrangeColor    = EGICreateColor(0xffff, 0x8000, 0);
  pEGraph->YellowColor    = EGICreateColor(0xffff, 0xffff, 0);
  pEGraph->TurquoiseColor = EGICreateColor(0, 0xffff, 0xffff);
  pEGraph->PurpleColor    = EGICreateColor(0xffff, 0, 0xffff);

  /* Load font to draw text */
  pEGraph->TextFont = gdk_font_load("-*-courier-medium-r-*-*-12-*-*-*-m-*-*-*");

  /* Initialize draw start time */
  pEGraph->DrawStartTime = 0;

  /* Return the new widget to the caller */
  return pEGraph;
}

/******************************************************************
 * Function :
 * Description :
 * Parameters :
 * Return values :
 * History :
 * Note :
 ******************************************************************/
void EGDisplayEventTrace(eventGraph* pmEventGraph, system* pmSystem, db* pmEventDB)
{
  /* Miscallaneous variables */
  gint             lNbProc;                 /* Number of processes in the system */
  gchar            lLabelString[270];       /* Label string */

  /* Process variables */
  process*         pProc;                   /* Generic process pointer */
  process*         pProcFirst;              /* First process in list */

  /* Time and size variables */
  struct timeval   lTime;                   /* The duration of the trace */

  /* Gtk widgets */
  GtkWidget*       pItemLabel;              /* Label of item */
  GtkWidget*       pListItem;               /* Item to add to list */
  GtkRequisition   lRequisition;            /* Used to obtain widget sizes*/

  /* Set data */
  pmEventGraph->EventDB = pmEventDB;
  pmEventGraph->System  = pmSystem;

  /* Can we draw anything */
  if((test_bit(TRACE_SYSCALL_ENTRY, &(pmEventGraph->EventDB->EventMask))
    &&test_bit(TRACE_SYSCALL_EXIT,  &(pmEventGraph->EventDB->EventMask))
    &&test_bit(TRACE_TRAP_ENTRY,    &(pmEventGraph->EventDB->EventMask))
    &&test_bit(TRACE_TRAP_EXIT,     &(pmEventGraph->EventDB->EventMask))
    &&test_bit(TRACE_IRQ_ENTRY,     &(pmEventGraph->EventDB->EventMask))
    &&test_bit(TRACE_IRQ_EXIT,      &(pmEventGraph->EventDB->EventMask))
    &&test_bit(TRACE_SCHEDCHANGE,   &(pmEventGraph->EventDB->EventMask))
    &&test_bit(TRACE_SYSCALL_ENTRY, &(pmEventGraph->EventDB->DetailsMask))
    &&test_bit(TRACE_SYSCALL_EXIT,  &(pmEventGraph->EventDB->DetailsMask))
    &&test_bit(TRACE_TRAP_ENTRY,    &(pmEventGraph->EventDB->DetailsMask))
    &&test_bit(TRACE_TRAP_EXIT,     &(pmEventGraph->EventDB->DetailsMask))
    &&test_bit(TRACE_IRQ_ENTRY,     &(pmEventGraph->EventDB->DetailsMask))
    &&test_bit(TRACE_IRQ_EXIT,      &(pmEventGraph->EventDB->DetailsMask))
    &&test_bit(TRACE_SCHEDCHANGE,   &(pmEventGraph->EventDB->DetailsMask))) != 1)
    {
    WDStatusBarDisplay(pmEventGraph->MainWindow,
		       "Insufficient trace information to draw graph");
    return;
    }

  /* Do we have multiple CPUs */
  if(pmEventGraph->EventDB->LogCPUID == TRUE)
    WDStatusBarDisplay(pmEventGraph->MainWindow,
		       "Displaying only the information for CPU #0");

#if 0
  g_print("Start Time : (%ld, %ld) \n", pmEventDB->StartTime.tv_sec, pmEventDB->StartTime.tv_usec);
  g_print("First Event : (%ld, %ld) \n", pmEventDB->Events->Time.tv_sec, pmEventDB->Events->Time.tv_usec);
  g_print("Total Nb Events : %d \n", pmEventDB->Nb);
#endif

  /* Find trace start time */
  pmEventGraph->StartTime = DBGetTimeInDouble(pmEventDB->DrawStartTime);
  
  /* Find trace duration */
  DBTimeSub(lTime, pmEventDB->EndTime, pmEventDB->DrawStartTime);

  /* Find out the duration in microseconds */
  pmEventGraph->Duration = DBGetTimeInDouble(lTime);

  /* Initialize drawable time interval */
  pmEventGraph->Interval = 0;

  /* Add a bogus item in order to properly draw topmost item */
  pListItem  = gtk_list_item_new_with_label("");
  gtk_container_add(GTK_CONTAINER(pmEventGraph->List), pListItem);
  gtk_widget_show(pListItem);
  gtk_signal_connect (GTK_OBJECT(pListItem), 
		      "select",
		      GTK_SIGNAL_FUNC(EGSHSelectListItem),
		      pmEventGraph);

  /* Display the processes in the process list */
  lNbProc = 1;
  pProcFirst = NULL;
  for(pProc = pmSystem->Processes; pProc != NULL; pProc = pProc->Next)
    {
    /* Keep count on the number of processes */
    lNbProc++;

    /* Create list item */
    if(pProc->Command != NULL)
      sprintf(lLabelString, "%s (%d)", pProc->Command, pProc->PID);
    else
      {
      /* Is this the 0 process */
      if(pProc->PID == 0)
	{
	pProcFirst = pProc;
	continue;
	}
      else
	sprintf(lLabelString, "Unnamed child (%d)", pProc->PID);
      }
    pListItem  = gtk_list_item_new_with_label(lLabelString);

    /* Put pointer to process in label */
    gtk_object_set_data(GTK_OBJECT(pListItem), gProcessItemData, pProc);

    /* Attach Signal handler */
    gtk_signal_connect (GTK_OBJECT(pListItem), 
			"select",
			GTK_SIGNAL_FUNC(EGSHSelectListItem),
			pmEventGraph);

    /* Show the label */
    gtk_widget_show(pListItem);
    
    /* Append item into list */
    gtk_container_add(GTK_CONTAINER(pmEventGraph->List), pListItem);

    /* Set item position in list */
    pProc->ListPos = gtk_list_child_position(GTK_LIST(pmEventGraph->List), pListItem);
    }

  /* Put first process */
  if(pProcFirst != NULL)
    {
    /* Create new label */
    sprintf(lLabelString, "Kernel (%d)", pProcFirst->PID);
    pListItem  = gtk_list_item_new_with_label(lLabelString);
    gtk_signal_connect (GTK_OBJECT(pListItem), 
			"select",
			GTK_SIGNAL_FUNC(EGSHSelectListItem),
			pmEventGraph);

    /* Put pointer to process in label */
    gtk_object_set_data(GTK_OBJECT(pListItem), gProcessItemData, pProcFirst);

    /* Show the label */
    gtk_widget_show(pListItem);

    /* Append item into list */
    gtk_container_add(GTK_CONTAINER(pmEventGraph->List), pListItem);

    /* Set item position in list */
    pProcFirst->ListPos = gtk_list_child_position(GTK_LIST(pmEventGraph->List), pListItem);
    }

  /* Set number of processes */
  pmEventGraph->NbProc = lNbProc;

  /* Get the height of a label */
  gtk_widget_size_request(GTK_WIDGET(pListItem), &lRequisition);

  /* Set the label height */
  pmEventGraph->LabelHeight = lRequisition.height;

  /* Set the number of microseconds per pixel and pixels for start */
  pmEventGraph->USecPerPix = EGRAPH_USEC_PER_PIX;
  pmEventGraph->PixPerUSec = 0;
  pmEventGraph->PixStart = EGRAPH_PIX_START;

  /* Set size of image as if all events where displayed in the same pixmap */
  if(pmEventGraph->USecPerPix > 0)
    pmEventGraph->HTrueSize = ((gint) pmEventGraph->Duration) / pmEventGraph->USecPerPix;
  else
    pmEventGraph->HTrueSize = ((gint) pmEventGraph->Duration)  * (gint)  (1 / pmEventGraph->USecPerPix);
  pmEventGraph->VTrueSize = pmEventGraph->LabelHeight * pmEventGraph->NbProc;

  /* Initialize draw sizes */
  pmEventGraph->HDrawSize = 0;
  pmEventGraph->VDrawSize = 0;

  /* Set start draw time and event */
  memset(&(pmEventGraph->DrawStartEvent), 0 , sizeof(pmEventGraph->DrawStartEvent));
  memset(&(pmEventGraph->DrawEndEvent), 0 , sizeof(pmEventGraph->DrawEndEvent));
  pmEventGraph->DrawStartTime  = pmEventGraph->StartTime;

  /* Initialize scrollbar position */
  pmEventGraph->LastScrollVal = 0;
  
  /* We have finished initializing the graph */
  pmEventGraph->Init = TRUE;
}

/******************************************************************
 * Function :
 * Description :
 * Parameters :
 * Return values :
 * History :
 * Note :
 ******************************************************************/
void EGClearEventGraph(eventGraph* pmEGraph)
{
  /* Reset widget state */
  pmEGraph->Init = FALSE;
  pmEGraph->SelectedProc = NULL;

  /* Reset data */
  pmEGraph->EventDB = NULL;
  pmEGraph->System  = 0;

  /* Reset the draw variables */
  pmEGraph->NbProc      = 0;
  pmEGraph->LabelHeight = 0;
  pmEGraph->PixStart    = 0;
  pmEGraph->PixPerUSec  = 0;
  pmEGraph->USecPerPix  = 0;
  pmEGraph->HTrueSize   = 0;
  pmEGraph->VTrueSize   = 0;
  pmEGraph->HDrawSize   = 0;
  pmEGraph->VDrawSize   = 0;
  pmEGraph->StartTime   = 0;
  pmEGraph->Duration    = 0;
  pmEGraph->Interval    = 0;

  /* Reset scroll information */
  memset(&(pmEGraph->DrawStartEvent), 0 , sizeof(pmEGraph->DrawStartEvent));
  memset(&(pmEGraph->DrawEndEvent), 0 , sizeof(pmEGraph->DrawEndEvent));
  pmEGraph->DrawStartTime  = 0;
  pmEGraph->LastScrollVal  = 0;

  /* Empty event icon coordinates list */
  if(pmEGraph->EventIconPos)
    {
    /* Free all allocated coordinate items */
    g_list_foreach(pmEGraph->EventIconPos, (GFunc) g_free, NULL);

    /* Free the GList list items */
    g_list_free(pmEGraph->EventIconPos);

    /* Reset list pointer */
    pmEGraph->EventIconPos = NULL;
    }

  /* Remove the items in the process list */
  gtk_list_clear_items(GTK_LIST(pmEGraph->List), 0, -1);

  /* Reset the draw area size */
  gtk_widget_set_usize(pmEGraph->DrawArea,
		       0,
		       0);

  /* Reset the ruler */
  gtk_ruler_set_range(GTK_RULER(pmEGraph->HRuler), 0, 0, 0, 0);

  /* Reset the horizontal scrollbar */
  GTK_ADJUSTMENT(pmEGraph->HAdjustment)->lower = 0;
  GTK_ADJUSTMENT(pmEGraph->HAdjustment)->upper = 0;
  GTK_ADJUSTMENT(pmEGraph->HAdjustment)->value = 0 ;
  GTK_ADJUSTMENT(pmEGraph->HAdjustment)->page_size = 0;
  GTK_ADJUSTMENT(pmEGraph->HAdjustment)->step_increment = 0;
  GTK_ADJUSTMENT(pmEGraph->HAdjustment)->page_increment = 0;
  gtk_adjustment_changed(GTK_ADJUSTMENT(pmEGraph->HAdjustment));

  /* Make sure changes to the draw area are done */
  /* IF THIS ISN'T DONE, THE DRAWING SOMEWHAT SOMETIMES DOESN'T GET ERASED ??? */
  gtk_widget_hide(pmEGraph->ScrolledDraw);
  gtk_widget_show(pmEGraph->ScrolledDraw);
}

/******************************************************************
 * Function :
 * Description :
 * Parameters :
 * Return values :
 * History :
 * Note :
 ******************************************************************/
void EGDestroyEventGraph(eventGraph** pmEGraph)
{

  /* Has the graph already been freed? */
  if(!pmEGraph) return;
  if(!*pmEGraph) return;

  /* Clear the event graph */
  EGClearEventGraph(*pmEGraph);

  /* Was there a pixmap already */
  if((*pmEGraph)->PixMap != NULL)
    {
    /* Free the already existing pixmap */
    gdk_pixmap_unref((*pmEGraph)->PixMap);
    (*pmEGraph)->PixMap = NULL;
    }

  /* Did we have cached icons? */
  if((*pmEGraph)->KernelTimer.Pixmap != NULL)
    {
    /* Free the already existing icons */
    gdk_pixmap_unref((*pmEGraph)->KernelTimer.Pixmap);
    gdk_pixmap_unref((*pmEGraph)->SysCall.Pixmap);
    gdk_pixmap_unref((*pmEGraph)->Trap.Pixmap);
    gdk_pixmap_unref((*pmEGraph)->SchedChange.Pixmap);
    gdk_pixmap_unref((*pmEGraph)->BottomHalf.Pixmap);
    gdk_pixmap_unref((*pmEGraph)->IOStart.Pixmap);
    gdk_pixmap_unref((*pmEGraph)->IOEnd.Pixmap);
    gdk_pixmap_unref((*pmEGraph)->IRQ.Pixmap);

    gdk_bitmap_unref((*pmEGraph)->KernelTimer.Mask);
    gdk_bitmap_unref((*pmEGraph)->SysCall.Mask);
    gdk_bitmap_unref((*pmEGraph)->Trap.Mask);
    gdk_bitmap_unref((*pmEGraph)->SchedChange.Mask);
    gdk_bitmap_unref((*pmEGraph)->BottomHalf.Mask);
    gdk_bitmap_unref((*pmEGraph)->IOStart.Mask);
    gdk_bitmap_unref((*pmEGraph)->IOEnd.Mask);
    gdk_bitmap_unref((*pmEGraph)->IRQ.Mask);

    (*pmEGraph)->KernelTimer.Pixmap = NULL;
    (*pmEGraph)->SysCall.Pixmap     = NULL;
    (*pmEGraph)->Trap.Pixmap        = NULL;
    (*pmEGraph)->SchedChange.Pixmap = NULL;
    (*pmEGraph)->BottomHalf.Pixmap  = NULL;
    (*pmEGraph)->IOStart.Pixmap     = NULL;
    (*pmEGraph)->IOEnd.Pixmap       = NULL;
    (*pmEGraph)->IRQ.Pixmap         = NULL;
    }

  /* Free allocated GDK structures */

  /* Fonts */
  gdk_font_unref((*pmEGraph)->TextFont);

  /* Colors */
  g_free((*pmEGraph)->RedColor);
  g_free((*pmEGraph)->BlueColor); 
  g_free((*pmEGraph)->GreyColor);
  g_free((*pmEGraph)->GreenColor); 
  g_free((*pmEGraph)->BlackColor); 
  g_free((*pmEGraph)->WhiteColor);
  g_free((*pmEGraph)->OrangeColor);
  g_free((*pmEGraph)->YellowColor);
  g_free((*pmEGraph)->TurquoiseColor);
  g_free((*pmEGraph)->PurpleColor);

  /* The GCs, if any were allocated */
  if((*pmEGraph)->ColorsAllocated == TRUE)
    {
    gdk_gc_destroy((*pmEGraph)->BackgroundGC);
    gdk_gc_destroy((*pmEGraph)->HorizonGC);
    gdk_gc_destroy((*pmEGraph)->ProcessGC);
    gdk_gc_destroy((*pmEGraph)->Process0GC);
    gdk_gc_destroy((*pmEGraph)->KernelGC);
    gdk_gc_destroy((*pmEGraph)->SelectedGC);
    gdk_gc_destroy((*pmEGraph)->SysCallGC);
    gdk_gc_destroy((*pmEGraph)->TrapGC);
    gdk_gc_destroy((*pmEGraph)->InterruptGC);
    gdk_gc_destroy((*pmEGraph)->TextGC);
    }

  /* destroy the structure */
  g_free(*pmEGraph);

  /* reset the structure pointer */
  *pmEGraph = NULL;
}
