/*
   GotoEvent.c : Go to event dialog window.
   Copyright (C) 1999 Jean-Hugues Deschenes.
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

   History : 
      JH.D., 11/09/1999, Initial typing.
  
   Note : The current version of this toolkit is for linux 2.2.0.
*/

#include "GotoEvent.h"

/**********************************************************************************/
/**************************** Signal handling functions ***************************/
/**********************************************************************************/
/******************************************************************
 * Function :
 * Description :
 * Parameters :
 * Return values :
 * History :
 * Note :
 ******************************************************************/
gint GTESHDeleteEvent(GtkWidget* pmWidget, GdkEvent* pmEvent, gpointer pmData)
{
  /* Close it */
  return(FALSE);
}

/******************************************************************
 * Function :
 * Description :
 * Parameters :
 * Return values :
 * History :
 * Note :
 ******************************************************************/
void GTESHDestroy(GtkWidget* pmWidget, gpointer pmData)
{
  gotoEventWindow*  pGTEWin;  /* The window for whom this was called */

  /* Do we have anything meaningfull */
  if((pGTEWin = (gotoEventWindow*) pmData) == NULL)
    {
    g_print("Internal error: Callback without View time frame Window \n");
    exit(1);
    }

  /* Call on the cancel callback */
  pGTEWin->CBCancel(pGTEWin->SysView);
}

/******************************************************************
 * Function :
 * Description :
 * Parameters :
 * Return values :
 * History :
 * Note :
 ******************************************************************/
void GTESHInternalOK(GtkWidget* pmWidget, gpointer pmData)
{
  gotoEventWindow*  pGTEWin;  /* The window for whom this was called */

  /* Do we have anything meaningfull */
  if((pGTEWin = (gotoEventWindow*) pmData) == NULL)
    {
    g_print("Internal error: Callback without Go to event Window \n");
    exit(1);
    }

  /* Set the appropriate query field */
  switch(pGTEWin->SearchParameters.SearchType)
    {
    case GOTO_EVENT_DESC:
      pGTEWin->SearchParameters.SearchString = 
	gtk_entry_get_text(GTK_ENTRY(pGTEWin->EventDescEntry));
      break;

    case GOTO_CPU_ID:
      pGTEWin->SearchParameters.CPUID = 
	atoi(gtk_entry_get_text(GTK_ENTRY(pGTEWin->CPUIDEntry)));
      break;

    case GOTO_EVENT_TYPE:
      /* Is set by the option menu callback */
      break;

    case GOTO_PID:
      pGTEWin->SearchParameters.PID = 
	atoi(gtk_entry_get_text(GTK_ENTRY(pGTEWin->PIDEntry)));
      break;

    case GOTO_ENTRY_LENGTH:
      pGTEWin->SearchParameters.EntryLength = 
	atoi(gtk_entry_get_text(GTK_ENTRY(pGTEWin->LengthEntry)));

    }


  /* Call the OK Callback */
  pGTEWin->CBOK(pGTEWin->SysView, pGTEWin->SearchParameters);
}

/******************************************************************
 * Function :
 *    GTESHSelectRadioButton()
 * Description :
 * Parameters :
 * Return values :
 * History :
 * Note :
 ******************************************************************/
void GTESHSelectRadioButton(GtkWidget* pmWidget, gpointer pmData)
{
  gotoEventWindow* pGTEWin; /* The dump to file window */

  if(!(pGTEWin = (gotoEventWindow*) pmData))
    {
    g_print("Selected radio without passing go to event window! \n");
    exit(1);
    }

  /* Make all entries insensitive */
  gtk_widget_set_sensitive(pGTEWin->CPUIDEntry,       FALSE);
  gtk_widget_set_sensitive(pGTEWin->EventTypeOptMenu, FALSE);
  gtk_widget_set_sensitive(pGTEWin->PIDEntry,         FALSE);
  gtk_widget_set_sensitive(pGTEWin->LengthEntry,      FALSE);
  gtk_widget_set_sensitive(pGTEWin->EventDescEntry,   FALSE);

  /* Set the appropriate search type and set entry active*/
  if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(pGTEWin->CPUIDRadio)))
    {
    pGTEWin->SearchParameters.SearchType = GOTO_CPU_ID;
    gtk_widget_set_sensitive(pGTEWin->CPUIDEntry, TRUE);
    }

  if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(pGTEWin->EventTypeRadio)))
    {
    pGTEWin->SearchParameters.SearchType = GOTO_EVENT_TYPE;
    gtk_widget_set_sensitive(pGTEWin->EventTypeOptMenu, TRUE);
    }

  if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(pGTEWin->PIDRadio)))
    {
    pGTEWin->SearchParameters.SearchType = GOTO_PID;
    gtk_widget_set_sensitive(pGTEWin->PIDEntry, TRUE);
    }

  if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(pGTEWin->LengthRadio)))
    {
    pGTEWin->SearchParameters.SearchType = GOTO_ENTRY_LENGTH;
    gtk_widget_set_sensitive(pGTEWin->LengthEntry, TRUE);
    }

  if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(pGTEWin->EventDescRadio)))
    {
    pGTEWin->SearchParameters.SearchType = GOTO_EVENT_DESC;
    gtk_widget_set_sensitive(pGTEWin->EventDescEntry, TRUE);
    }

}

/******************************************************************
 * Function :
 *    GTESHSelectEventType()
 * Description :
 * Parameters :
 * Return values :
 * History :
 * Note :
 ******************************************************************/
void GTESHSelectEventType(GtkWidget* pmWidget, gpointer pmData)
{
  GTESearchQuery* lSearchParams; /* Search parameters */

  if(!(lSearchParams = (GTESearchQuery*) pmData))
    {
    g_print("Selected event type without passing search parameters! \n");
    exit(1);
    }

  /* retireve the event type sought */
  lSearchParams->EventType = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(pmWidget), EVENT_TYPE_MENU_DATA));
}

/**********************************************************************************/
/******************************** Internal functions ******************************/
/**********************************************************************************/
/******************************************************************
 * Function :
 * Description :
 * Parameters :
 * Return values :
 * History :
 * Note :
 ******************************************************************/
void GTEIAddMenuItem(GtkWidget*      pmMenu,
		     GTESearchQuery* SearchParams,
		     gchar*          MenuText,
		     gint            TypeID)
{
  GtkWidget* NewMenuItem; /* The new menu item to add */

  /* Create the new menu item */
  NewMenuItem = gtk_menu_item_new_with_label(MenuText);

  /* Append the menu item to the menu */
  gtk_menu_append(GTK_MENU(pmMenu), NewMenuItem);

  /* Memorize the associated Event Type ID */
  gtk_object_set_data(GTK_OBJECT(NewMenuItem), EVENT_TYPE_MENU_DATA, GINT_TO_POINTER(TypeID));

  /* Connect selection signal */
  gtk_signal_connect(GTK_OBJECT(NewMenuItem),
		     "activate",
		     GTK_SIGNAL_FUNC(GTESHSelectEventType),
		     SearchParams);

  /* Make the menu item visible */
  gtk_widget_show(NewMenuItem);
}

/**********************************************************************************/
/******************************* Windowing functions ******************************/
/**********************************************************************************/
/******************************************************************
 * Function :
 * Description :
 * Parameters :
 * Return values :
 * History :
 * Note :
 ******************************************************************/
void GTEConnectSignals(gotoEventWindow* pmGTEWin)
{
  /* Connect the generic signals */
  gtk_signal_connect(GTK_OBJECT(pmGTEWin->Window),
		     "delete_event",
		     GTK_SIGNAL_FUNC(GTESHDeleteEvent),
		     pmGTEWin);
  gtk_signal_connect(GTK_OBJECT(pmGTEWin->Window),
		     "destroy",
		     GTK_SIGNAL_FUNC(GTESHDestroy),
		     pmGTEWin);

  /* Connect the radio button signals */
  gtk_signal_connect(GTK_OBJECT(pmGTEWin->CPUIDRadio),
		     "toggled",
		     GTK_SIGNAL_FUNC(GTESHSelectRadioButton),
		     pmGTEWin);
  gtk_signal_connect(GTK_OBJECT(pmGTEWin->EventTypeRadio),
		     "toggled",
		     GTK_SIGNAL_FUNC(GTESHSelectRadioButton),
		     pmGTEWin);
  gtk_signal_connect(GTK_OBJECT(pmGTEWin->PIDRadio),
		     "toggled",
		     GTK_SIGNAL_FUNC(GTESHSelectRadioButton),
		     pmGTEWin);
  gtk_signal_connect(GTK_OBJECT(pmGTEWin->LengthRadio),
		     "toggled",
		     GTK_SIGNAL_FUNC(GTESHSelectRadioButton),
		     pmGTEWin);
  gtk_signal_connect(GTK_OBJECT(pmGTEWin->EventDescRadio),
		     "toggled",
		     GTK_SIGNAL_FUNC(GTESHSelectRadioButton),
		     pmGTEWin);

  /* Connect the Ok and Cancel buttons */
  gtk_signal_connect(GTK_OBJECT(pmGTEWin->OKButton),
		     "clicked",
		     GTK_SIGNAL_FUNC(GTESHInternalOK),
		     pmGTEWin);
  gtk_signal_connect_object(GTK_OBJECT(pmGTEWin->CancelButton),
			    "clicked",
			    GTK_SIGNAL_FUNC(gtk_widget_destroy),
			    GTK_OBJECT(pmGTEWin->Window));
}

/******************************************************************
 * Function :
 * Description :
 * Parameters :
 * Return values :
 * History :
 * Note :
 ******************************************************************/
void GTEShowGotoEventWindow(gotoEventWindow* pmGTEWin)
{
  /* Make OK button default choice */
  gtk_widget_grab_default(pmGTEWin->OKButton);

  /* Show it all to the world */
  gtk_widget_show(pmGTEWin->MainTable);
  gtk_widget_show(pmGTEWin->SearchLabel);
  gtk_widget_show(pmGTEWin->CPUIDRadio);
  gtk_widget_show(pmGTEWin->EventTypeRadio);
  gtk_widget_show(pmGTEWin->PIDRadio);
  gtk_widget_show(pmGTEWin->LengthRadio);
  gtk_widget_show(pmGTEWin->EventDescRadio);
  gtk_widget_show(pmGTEWin->CPUIDEntry);
  gtk_widget_show(pmGTEWin->PIDEntry);
  gtk_widget_show(pmGTEWin->LengthEntry);
  gtk_widget_show(pmGTEWin->EventDescEntry);
  gtk_widget_show(pmGTEWin->EventTypeOptMenu);
  gtk_widget_show(pmGTEWin->EventTypeListMenu);
  gtk_widget_show(pmGTEWin->OKButton);
  gtk_widget_show(pmGTEWin->CancelButton);

  /* Make Event description default search and deactivate all other entries through callback */
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(pmGTEWin->EventDescRadio), TRUE);

  /* Set title */
  gtk_window_set_title(GTK_WINDOW(pmGTEWin->Window), "Go to event...");

  /* Display in the center of screen */
  gtk_window_set_position(GTK_WINDOW(pmGTEWin->Window), GTK_WIN_POS_CENTER);

  /* Make window unshrinkable and always visible */
  gtk_window_set_policy(GTK_WINDOW(pmGTEWin->Window), FALSE, TRUE, FALSE);
  gtk_window_set_transient_for(GTK_WINDOW(pmGTEWin->Window), 
			       GTK_WINDOW(pmGTEWin->ParentWin));

  /* Display it */
  gtk_widget_show(pmGTEWin->Window);
}

/******************************************************************
 * Function :
 * Description :
 * Parameters :
 * Return values :
 * History :
 * Note :
 ******************************************************************/
gotoEventWindow* GTECreateGotoEventWindow(gpointer      pmSysView,
					  GtkWidget*    pmParentWin,
					  _GTECBOK*     pmCBOK,
					  _GTECBCancel* pmCBCancel)
{
  gotoEventWindow* pGTEWin;      /* Go to event window */

  /* Create a new trace file selection window */
  pGTEWin = (gotoEventWindow*) g_malloc(sizeof(gotoEventWindow));

  /* Put the given data in the right place */
  pGTEWin->SysView  = pmSysView;
  pGTEWin->ParentWin= pmParentWin;
  pGTEWin->CBOK     = pmCBOK;
  pGTEWin->CBCancel = pmCBCancel;

  /* Build the window */
  pGTEWin->Window     = gtk_window_new(GTK_WINDOW_DIALOG);
  pGTEWin->MainTable  = gtk_table_new(7, 2, FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(pGTEWin->Window), 5);
  gtk_container_add(GTK_CONTAINER(pGTEWin->Window), pGTEWin->MainTable);

  /* Create the label and pack it */
  pGTEWin->SearchLabel = gtk_label_new("Search for: ");
  gtk_table_attach_defaults(GTK_TABLE(pGTEWin->MainTable), pGTEWin->SearchLabel,
			    0, 2, 0, 1);

  /* Create Radio buttons and pack them */
  pGTEWin->RadioGroup     = NULL;
  pGTEWin->CPUIDRadio     = gtk_radio_button_new_with_label(pGTEWin->RadioGroup, "CPU ID:");
  pGTEWin->RadioGroup     = gtk_radio_button_group(GTK_RADIO_BUTTON(pGTEWin->CPUIDRadio));
  pGTEWin->EventTypeRadio = gtk_radio_button_new_with_label(pGTEWin->RadioGroup, "Event type:");
  pGTEWin->RadioGroup     = gtk_radio_button_group(GTK_RADIO_BUTTON(pGTEWin->EventTypeRadio));
  pGTEWin->PIDRadio       = gtk_radio_button_new_with_label(pGTEWin->RadioGroup, "PID:");
  pGTEWin->RadioGroup     = gtk_radio_button_group(GTK_RADIO_BUTTON(pGTEWin->PIDRadio));
  pGTEWin->LengthRadio    = gtk_radio_button_new_with_label(pGTEWin->RadioGroup, "Entry Length:");
  pGTEWin->RadioGroup     = gtk_radio_button_group(GTK_RADIO_BUTTON(pGTEWin->LengthRadio));
  pGTEWin->EventDescRadio = gtk_radio_button_new_with_label(pGTEWin->RadioGroup, "Event Description:");
  pGTEWin->RadioGroup     = gtk_radio_button_group(GTK_RADIO_BUTTON(pGTEWin->EventDescRadio));
  gtk_table_attach_defaults(GTK_TABLE(pGTEWin->MainTable), pGTEWin->CPUIDRadio,
			    0, 1, 1, 2);
  gtk_table_attach_defaults(GTK_TABLE(pGTEWin->MainTable), pGTEWin->EventTypeRadio,
			    0, 1, 2, 3);
  gtk_table_attach_defaults(GTK_TABLE(pGTEWin->MainTable), pGTEWin->PIDRadio,
			    0, 1, 3, 4);
  gtk_table_attach_defaults(GTK_TABLE(pGTEWin->MainTable), pGTEWin->LengthRadio,
			    0, 1, 4, 5);
  gtk_table_attach_defaults(GTK_TABLE(pGTEWin->MainTable), pGTEWin->EventDescRadio,
			    0, 1, 5, 6);

  /* Create search entries and pack them */
  pGTEWin->CPUIDEntry     = gtk_entry_new();
  pGTEWin->PIDEntry       = gtk_entry_new();
  pGTEWin->LengthEntry    = gtk_entry_new();
  pGTEWin->EventDescEntry = gtk_entry_new();
  gtk_table_attach_defaults(GTK_TABLE(pGTEWin->MainTable), pGTEWin->CPUIDEntry,
			    1, 2, 1, 2);
  gtk_table_attach_defaults(GTK_TABLE(pGTEWin->MainTable), pGTEWin->PIDEntry,
			    1, 2, 3, 4);
  gtk_table_attach_defaults(GTK_TABLE(pGTEWin->MainTable), pGTEWin->LengthEntry,
			    1, 2, 4, 5);
  gtk_table_attach_defaults(GTK_TABLE(pGTEWin->MainTable), pGTEWin->EventDescEntry,
			    1, 2, 5, 6);

  /* Create Event Type option menu and pack it */
  pGTEWin->EventTypeOptMenu  = gtk_option_menu_new();
  pGTEWin->EventTypeListMenu = gtk_menu_new();
  gtk_table_attach_defaults(GTK_TABLE(pGTEWin->MainTable), pGTEWin->EventTypeOptMenu,
			    1, 2, 2, 3);

  /* Populate the option menu's associated menu */
  GTEIAddMenuItem(pGTEWin->EventTypeListMenu, &pGTEWin->SearchParameters,
		  "Trace start",                 TRACE_START);
  GTEIAddMenuItem(pGTEWin->EventTypeListMenu, &pGTEWin->SearchParameters,
		  "System call entry",           TRACE_SYSCALL_ENTRY);
  GTEIAddMenuItem(pGTEWin->EventTypeListMenu, &pGTEWin->SearchParameters,
		  "System call exit",            TRACE_SYSCALL_EXIT);
  GTEIAddMenuItem(pGTEWin->EventTypeListMenu, &pGTEWin->SearchParameters,
		  "Trap entry",                  TRACE_TRAP_ENTRY);
  GTEIAddMenuItem(pGTEWin->EventTypeListMenu, &pGTEWin->SearchParameters,
		  "Trap exit",                   TRACE_TRAP_EXIT);
  GTEIAddMenuItem(pGTEWin->EventTypeListMenu, &pGTEWin->SearchParameters,
		  "IRQ entry",                   TRACE_IRQ_ENTRY);
  GTEIAddMenuItem(pGTEWin->EventTypeListMenu, &pGTEWin->SearchParameters,
		  "IRQ exit",                    TRACE_IRQ_EXIT);
  GTEIAddMenuItem(pGTEWin->EventTypeListMenu, &pGTEWin->SearchParameters,
		  "Schedule change",             TRACE_SCHEDCHANGE);
  GTEIAddMenuItem(pGTEWin->EventTypeListMenu, &pGTEWin->SearchParameters,
		  "Kernel timer",                TRACE_KERNEL_TIMER);
  GTEIAddMenuItem(pGTEWin->EventTypeListMenu, &pGTEWin->SearchParameters,
		  "Bottom half",                 TRACE_BOTTOM_HALF);
  GTEIAddMenuItem(pGTEWin->EventTypeListMenu, &pGTEWin->SearchParameters,
		  "Process management",          TRACE_PROCESS);
  GTEIAddMenuItem(pGTEWin->EventTypeListMenu, &pGTEWin->SearchParameters,
		  "File system management",      TRACE_FILE_SYSTEM);
  GTEIAddMenuItem(pGTEWin->EventTypeListMenu, &pGTEWin->SearchParameters,
		  "Timer management",            TRACE_TIMER);
  GTEIAddMenuItem(pGTEWin->EventTypeListMenu, &pGTEWin->SearchParameters,
		  "Memory management",           TRACE_MEMORY);
  GTEIAddMenuItem(pGTEWin->EventTypeListMenu, &pGTEWin->SearchParameters,
		  "Socket communication", TRACE_SOCKET);
  GTEIAddMenuItem(pGTEWin->EventTypeListMenu, &pGTEWin->SearchParameters,
		  "Inter-process communication", TRACE_IPC);
  GTEIAddMenuItem(pGTEWin->EventTypeListMenu, &pGTEWin->SearchParameters,
		  "Network communication",       TRACE_NETWORK);

  /* Associate menu and option menu, set default value */
  gtk_option_menu_set_menu(GTK_OPTION_MENU(pGTEWin->EventTypeOptMenu), pGTEWin->EventTypeListMenu);
  gtk_option_menu_set_history(GTK_OPTION_MENU(pGTEWin->EventTypeOptMenu), 0);
  pGTEWin->SearchParameters.EventType = TRACE_START;

  /* Create OK and cancel buttons and pack them */
  pGTEWin->OKButton     = gtk_button_new_with_label("Find");
  pGTEWin->CancelButton = gtk_button_new_with_label("Close");
  GTK_WIDGET_SET_FLAGS(pGTEWin->OKButton, GTK_CAN_DEFAULT);  /* Allow OK button to be default */
  gtk_table_attach_defaults(GTK_TABLE(pGTEWin->MainTable), pGTEWin->OKButton,
			    0, 1, 6, 7);
  gtk_table_attach_defaults(GTK_TABLE(pGTEWin->MainTable), pGTEWin->CancelButton,
			    1, 2, 6, 7);

  /* Give it to the caller */
  return pGTEWin;
}

/******************************************************************
 * Function :
 * Description :
 * Parameters :
 * Return values :
 * History :
 * Note :
 ******************************************************************/
void GTEDestroyGotoEventWindow(gotoEventWindow* pmGTEWin)
{
  /* Destroy the window */
  gtk_widget_destroy(pmGTEWin->Window);

  /* Free space used by structure */
  g_free(pmGTEWin);
}
