/*
 * TraceToolkit.h
 *
 * Copyright (C) 1999 Karim Yaghmour.
 *
 * This is distributed under GPL.
 *
 * Header for trace toolkit.
 *
 * History : 
 *    K.Y., 09/09/1999, Modified to reduce data size
 *    K.Y., 26/06/1999, Initial typing.
 *
 * Note : The current version of this toolkit is for linux 2.2.0.
 */

#ifndef __TRACE_TOOLKIT_MAIN_HEADER__
#define __TRACE_TOOLKIT_MAIN_HEADER__

#include <sys/types.h>
#include <asm/bitops.h>

/* Some type corrections, just in case */
#ifndef uint8_t
#define uint8_t u_int8_t
#endif
#ifndef uint16_t
#define uint16_t u_int16_t
#endif
#ifndef uint32_t
#define uint32_t u_int32_t
#endif

/* Linux 2.2.0 needed declarations */
struct timeval {
	long		tv_sec;		/* seconds */
        long      	tv_usec;	/* microseconds */
};

/* Trace mask */
typedef uint32_t trace_event_mask;

/* Boolean stuff */
#define TRUE  1
#define FALSE 0

/* Traced events */
#define TRACE_START           0    /* This is to mark the trace's start */
#define TRACE_SYSCALL_ENTRY   1    /* Entry in a given system call */
#define TRACE_SYSCALL_EXIT    2    /* Exit from a given system call */
#define TRACE_TRAP_ENTRY      3    /* Entry in a trap */
#define TRACE_TRAP_EXIT       4    /* Exit from a trap */
#define TRACE_IRQ_ENTRY       5    /* Entry in an irq */
#define TRACE_IRQ_EXIT        6    /* Exit from an irq */
#define TRACE_SCHEDCHANGE     7    /* Scheduling change */
#define TRACE_KERNEL_TIMER    8    /* The kernel timer routine has been called */
#define TRACE_BOTTOM_HALF     9    /* Going to run a bottom half */
#define TRACE_PROCESS        10    /* Hit key part of process management */
#define TRACE_FILE_SYSTEM    11    /* Hit key part of file system */
#define TRACE_TIMER          12    /* Hit key part of timer management */
#define TRACE_MEMORY         13    /* Hit key part of memory management */
#define TRACE_SOCKET         14    /* Hit key part of socket communication */
#define TRACE_IPC            15    /* Hit key part of inter-process communication */
#define TRACE_NETWORK        16    /* Hit key part of network communication */

#define TRACE_BUFFER_START   17    /* Mark the begining of a trace buffer */
#define TRACE_BUFFER_END     18    /* Mark the ending of a trace buffer */
#define TRACE_NEW_EVENT      19    /* New event type */
#define TRACE_CUSTOM         20    /* Custom event */

/* Number of traced events */
#define TRACE_MAX            TRACE_CUSTOM

/* Structures and macros for events */
/* TRACE_START */
#define TRACER_MAGIC_NUMBER         0x000D6B7ED     /* That day marks an important historical event ... */
#define TRACER_SUP_VERSION_MAJOR    1               /* Major supported version number */
#define TRACER_SUP_VERSION_MINOR    3               /* Minor supported version number */
typedef struct _trace_start
{
  uint32_t           MagicNumber;          /* Magic number to identify a trace */
  uint8_t            MajorVersion;         /* Major version of trace */
  uint8_t            MinorVersion;         /* Minor version of trace */
  uint32_t           BufferSize;           /* Size of buffers */
  trace_event_mask   EventMask;            /* The event mask */
  trace_event_mask   DetailsMask;          /* Are the event details logged */
  uint8_t            LogCPUID;             /* Is the CPUID logged */
} trace_start;
#define START_EVENT(X) ((trace_start*)X)

/*  TRACE_SYSCALL_ENTRY */
typedef struct _trace_syscall_entry
{
  uint8_t   syscall_id;  /* Syscall entry number in entry.S */
  uint32_t  address;     /* Address from which call was made */
} trace_syscall_entry;
#define SYSCALL_EVENT(X) ((trace_syscall_entry*)X)

/*  TRACE_TRAP_ENTRY */
typedef struct _trace_trap_entry
{
  uint8_t   trap_id;     /* Trap number */
  uint32_t  address;     /* Address where trap occured */
} trace_trap_entry;
#define TRAP_EVENT(X) ((trace_trap_entry*)X)

/*  TRACE_IRQ_ENTRY */
typedef struct _trace_irq_entry
{
  uint8_t  irq_id : 6;      /* IRQ number 2^6 = 64 */
  uint8_t  unused : 1;      /* This isn't used for now */
  uint8_t  kernel : 1;      /* Are we executing kernel code */
} trace_irq_entry;
#define IRQ_EVENT(X) ((trace_irq_entry*)X)

/*  TRACE_SCHEDCHANGE */ 
typedef struct _trace_schedchange
{
  int       out;         /* Outgoing process */
  int       in;          /* Incoming process */
  uint32_t  out_state;   /* Outgoing process' state */
} trace_schedchange;
#define SCHED_EVENT(X) ((trace_schedchange*)X)

/*  TRACE_BOTTOM_HALF */
#define trace_bottom_half uint8_t   /* ID of bottom half */
#define BH_EVENT(X) ((trace_bottom_half*)X)

/*  TRACE_PROCESS */
#define TRACE_PROCESS_KTHREAD     1  /* Creation of a kernel thread */
#define TRACE_PROCESS_FORK        2  /* A fork or clone occured */
#define TRACE_PROCESS_EXIT        3  /* An exit occured */
#define TRACE_PROCESS_WAIT        4  /* A wait occured */
#define TRACE_PROCESS_SIGNAL      5  /* A signal has been sent */
#define TRACE_PROCESS_WAKEUP      6  /* Wake up a process */
typedef struct _trace_process
{
  uint8_t   event_sub_id;    /* Process event ID */
  uint32_t  event_data1;     /* Data associated with event */
  uint32_t  event_data2;     
} trace_process;
#define PROC_EVENT(X) ((trace_process*)X)

/*  TRACE_FILE_SYSTEM */
#define TRACE_FILE_SYSTEM_BUF_WAIT_START  1  /* Starting to wait for a data buffer */
#define TRACE_FILE_SYSTEM_BUF_WAIT_END    2  /* End to wait for a data buffer */
#define TRACE_FILE_SYSTEM_EXEC            3  /* An exec occured */
#define TRACE_FILE_SYSTEM_OPEN            4  /* An open occured */
#define TRACE_FILE_SYSTEM_CLOSE           5  /* A close occured */
#define TRACE_FILE_SYSTEM_READ            6  /* A read occured */
#define TRACE_FILE_SYSTEM_WRITE           7  /* A write occured */
#define TRACE_FILE_SYSTEM_SEEK            8  /* A seek occured */
#define TRACE_FILE_SYSTEM_IOCTL           9  /* An ioctl occured */
#define TRACE_FILE_SYSTEM_SELECT         10  /* A select occured */
#define TRACE_FILE_SYSTEM_POLL           11  /* A poll occured */
typedef struct _trace_file_system
{
  uint8_t   event_sub_id;    /* File system event ID */
  uint32_t  event_data1;     /* Event data */
  uint32_t  event_data2;     /* Event data 2 */
  char*     file_name;       /* Name of file operated on */
} trace_file_system;
#define FS_EVENT(X) ((trace_file_system*)X)
#define FS_EVENT_FILENAME(X) ((char*) ((X) + sizeof(trace_file_system)))

/*  TRACE_TIMER */
#define TRACE_TIMER_EXPIRED      1  /* Timer expired */
#define TRACE_TIMER_SETITIMER    2  /* Setting itimer occurred */
#define TRACE_TIMER_SETTIMEOUT   3  /* Setting sched timeout occurred */
typedef struct _trace_timer
{
  uint8_t   event_sub_id;    /* Timer event ID */
  uint8_t   event_sdata;     /* Short data */
  uint32_t  event_data1;     /* Data associated with event */
  uint32_t  event_data2;     
} trace_timer;
#define TIMER_EVENT(X) ((trace_timer*)X)

/*  TRACE_MEMORY */
#define TRACE_MEMORY_PAGE_ALLOC        1  /* Allocating pages */
#define TRACE_MEMORY_PAGE_FREE         2  /* Freing pages */
#define TRACE_MEMORY_SWAP_IN           3  /* Swaping pages in */
#define TRACE_MEMORY_SWAP_OUT          4  /* Swaping pages out */
#define TRACE_MEMORY_PAGE_WAIT_START   5  /* Start to wait for page */
#define TRACE_MEMORY_PAGE_WAIT_END     6  /* End to wait for page */
typedef struct _trace_memory
{
  uint8_t        event_sub_id;    /* Memory event ID */
  unsigned long  event_data;      /* Data associated with event */
} trace_memory;
#define MEM_EVENT(X) ((trace_memory*)X)

/*  TRACE_SOCKET */
#define TRACE_SOCKET_CALL     1  /* A socket call occured */
#define TRACE_SOCKET_CREATE   2  /* A socket has been created */
#define TRACE_SOCKET_SEND     3  /* Data was sent to a socket */
#define TRACE_SOCKET_RECEIVE  4  /* Data was read from a socket */
typedef struct _trace_socket
{
  uint8_t   event_sub_id;    /* Socket event ID */
  uint32_t  event_data1;     /* Data associated with event */
  uint32_t  event_data2;     /* Data associated with event */
} trace_socket;
#define SOCKET_EVENT(X) ((trace_socket*)X)

/*  TRACE_IPC */
#define TRACE_IPC_CALL            1  /* A System V IPC call occured */
#define TRACE_IPC_MSG_CREATE      2  /* A message queue has been created */
#define TRACE_IPC_SEM_CREATE      3  /* A semaphore was created */
#define TRACE_IPC_SHM_CREATE      4  /* A shared memory segment has been created */
typedef struct _trace_ipc
{
  uint8_t   event_sub_id;    /* IPC event ID */
  uint32_t  event_data1;     /* Data associated with event */
  uint32_t  event_data2;     /* Data associated with event */
} trace_ipc;
#define IPC_EVENT(X) ((trace_ipc*)X)

/*  TRACE_NETWORK */
#define TRACE_NETWORK_PACKET_IN   1  /* A packet came in */
#define TRACE_NETWORK_PACKET_OUT  2  /* A packet was sent */
typedef struct _trace_network
{
  uint8_t  event_sub_id;   /* Network event ID */
  uint32_t event_data;     /* Event data */
} trace_network;
#define NET_EVENT(X) ((trace_network*)X)

/* Start and end of trace buffer information */
typedef struct _trace_buffer_start
{
  struct timeval     Time;                 /* Time stamp of this buffer */
  uint32_t           ID;                   /* Unique buffer ID */  
} trace_buffer_start;

/* Custom declared events */
typedef struct _trace_new_event
{
  uint16_t           ID;                    /* Unique event ID for new event */
  uint32_t           DescriptSize;          /* Size of the description of the event */
  char*              Description;           /* Description of new event */
} trace_new_event;
typedef struct _trace_custom
{
  uint16_t           ID;                    /* Event ID */
  uint32_t           DataSize;              /* Size of data recorded by event */
  void*              Data;                  /* Data recorded by event */
} trace_custom;

/* Options structure */
typedef struct _options
{
  int              Graphic;         /* Are we to go in graphic mode */
  int              Dump;            /* Do we only dump to output without interpreting anything */
  int              Omit;            /* Are we in "omit" mode */
  trace_event_mask OmitMask;        /* Mask with omitted events set to 1 */
  int              Trace;           /* Are we in trace mode */
  trace_event_mask TraceMask;       /* Mask with traced events set to 1 */
  int              TraceCPUID;      /* Trace only on CPUID */
  int              CPUID;           /* CPUID being traced */
  int              TracePID;        /* Trace only one PID */
  int              PID;             /* PID being traced */
  int              Summarize;       /* Is the summary requested */
  int              AcctSysCall;     /* Account time spent in system calls */
  int              ForgetCPUID;     /* Do we print the CPUID */
  int              ForgetTime;      /* Do we print time */
  int              ForgetPID;       /* Do we print the PID */
  int              ForgetDataLen;   /* Do we print length of data entry */
  int              ForgetString;    /* Do we print description string */
  char*            InputFileName;   /* Name of file used for input */
  char*            OutputFileName;  /* Name of file used for output */
  char*            ProcFileName;    /* Name of file contaning /proc information as of daemon's startup */
} options;

/* Functions */

options* CreateOptions
           (void);
void     DestroyOptions
           (options*  /* Options to be destroyed */);


#endif /* __TRACE_TOOLKIT_MAIN_HEADER__ */

