/*
   ViewTime.c : View time frame dialog window.
   Copyright (C) 1999 Jean-Hugues Deschenes.
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

   History : 
      JH.D., 11/09/1999, Initial typing.
  
   Note : The current version of this toolkit is for linux 2.2.0.
*/

#include "ViewTime.h"

/**********************************************************************************/
/**************************** Signal handling functions ***************************/
/**********************************************************************************/
/******************************************************************
 * Function :
 * Description :
 * Parameters :
 * Return values :
 * History :
 * Note :
 ******************************************************************/
gint VTFSHDeleteEvent(GtkWidget* pmWidget, GdkEvent* pmEvent, gpointer pmData)
{
  /* Close it */
  return(FALSE);
}

/******************************************************************
 * Function :
 * Description :
 * Parameters :
 * Return values :
 * History :
 * Note :
 ******************************************************************/
void VTFSHDestroy(GtkWidget* pmWidget, gpointer pmData)
{
  viewTimeFrameWindow*  pVTFWin;  /* The window for whom this was called */

  /* Do we have anything meaningfull */
  if((pVTFWin = (viewTimeFrameWindow*) pmData) == NULL)
    {
    g_print("Internal error: Callback without View time frame Window \n");
    exit(1);
    }

  /* Call on the cancel callback */
  pVTFWin->CBCancel(pVTFWin->SysView);
}

/******************************************************************
 * Function :
 * Description :
 * Parameters :
 * Return values :
 * History :
 * Note :
 ******************************************************************/
void VTFSHInternalOK(GtkWidget* pmWidget, gpointer pmData)
{
  viewTimeFrameWindow*  pVTFWin;  /* The window for whom this was called */

  /* Do we have anything meaningfull */
  if((pVTFWin = (viewTimeFrameWindow*) pmData) == NULL)
    {
    g_print("Internal error: Callback without View time frame Window \n");
    exit(1);
    }

  /* Call the OK Callback */
  pVTFWin->CBOK(pVTFWin->SysView,
		g_strtod(gtk_entry_get_text(GTK_ENTRY(pVTFWin->StartTimeEntry)), NULL),
		g_strtod(gtk_entry_get_text(GTK_ENTRY(pVTFWin->EndTimeEntry)), NULL));
}

/**********************************************************************************/
/******************************* Windowing functions ******************************/
/**********************************************************************************/
/******************************************************************
 * Function :
 * Description :
 * Parameters :
 * Return values :
 * History :
 * Note :
 ******************************************************************/
void VTFConnectSignals(viewTimeFrameWindow* pmVTFWin)
{
  /* Connect the generic signals */
  gtk_signal_connect(GTK_OBJECT(pmVTFWin->Window),
		     "delete_event",
		     GTK_SIGNAL_FUNC(VTFSHDeleteEvent),
		     pmVTFWin);
  gtk_signal_connect(GTK_OBJECT(pmVTFWin->Window),
		     "destroy",
		     GTK_SIGNAL_FUNC(VTFSHDestroy),
		     pmVTFWin);

  /* Connect the Ok and Cancel buttons */
  gtk_signal_connect(GTK_OBJECT(pmVTFWin->OKButton),
		     "clicked",
		     GTK_SIGNAL_FUNC(VTFSHInternalOK),
		     pmVTFWin);
  gtk_signal_connect_object(GTK_OBJECT(pmVTFWin->OKButton),
			    "clicked",
			    GTK_SIGNAL_FUNC(gtk_widget_destroy),
			    GTK_OBJECT(pmVTFWin->Window));
  gtk_signal_connect_object(GTK_OBJECT(pmVTFWin->CancelButton),
			    "clicked",
			    GTK_SIGNAL_FUNC(gtk_widget_destroy),
			    GTK_OBJECT(pmVTFWin->Window));
}

/******************************************************************
 * Function :
 * Description :
 * Parameters :
 * Return values :
 * History :
 * Note :
 ******************************************************************/
void VTFShowViewTimeFrameWindow(viewTimeFrameWindow* pmVTFWin)
{
  /* Make OK button default choice */
  gtk_widget_grab_default(pmVTFWin->OKButton);

  /* Show it all to the world */
  gtk_widget_show(pmVTFWin->VBox);
  gtk_widget_show(pmVTFWin->StartHBox);
  gtk_widget_show(pmVTFWin->EndHBox);
  gtk_widget_show(pmVTFWin->ButtonHBox);
  gtk_widget_show(pmVTFWin->TraceTimeLabel);
  gtk_widget_show(pmVTFWin->StartTimeLabel);
  gtk_widget_show(pmVTFWin->EndTimeLabel);
  gtk_widget_show(pmVTFWin->StartTimeEntry);
  gtk_widget_show(pmVTFWin->EndTimeEntry);
  gtk_widget_show(pmVTFWin->OKButton);
  gtk_widget_show(pmVTFWin->CancelButton);

  /* Set title */
  gtk_window_set_title(GTK_WINDOW(pmVTFWin->Window), "View time frame");

  /* Display in the center of screen */
  gtk_window_set_position(GTK_WINDOW(pmVTFWin->Window), GTK_WIN_POS_CENTER);

  /* Set the dialog window size */
/*  gtk_widget_set_usize(pmVTFWin->Window, 350, 70);*/

  /* Make OpenTrace window unresizeable and always visible */
  gtk_window_set_policy(GTK_WINDOW(pmVTFWin->Window),FALSE, FALSE, FALSE);
  gtk_window_set_transient_for(GTK_WINDOW(pmVTFWin->Window), 
			       GTK_WINDOW(pmVTFWin->ParentWin));

  /* Display it */
  gtk_widget_show(pmVTFWin->Window);
}

/******************************************************************
 * Function :
 * Description :
 * Parameters :
 * Return values :
 * History :
 * Note :
 ******************************************************************/
viewTimeFrameWindow* VTFCreateViewTimeFrameWindow(gpointer      pmSysView,
						  GtkWidget*    pmParentWin,
						  eventGraph*   pmEGraph,
						  _VTFCBOK*     pmCBOK,
						  _VTFCBCancel* pmCBCancel)
{
  gchar                lString[80];  /* Temporary string to build labels */
  viewTimeFrameWindow* pVTFWin;      /* View Time Frame window */

  if(!pmEGraph)
    {
    g_print("Internal error: cannot create View time frame window with NULL eventGraph! \n");
    exit(1);
    }

  /* Create a new trace file selection window */
  pVTFWin = (viewTimeFrameWindow*) g_malloc(sizeof(viewTimeFrameWindow));

  /* Put the given data in the right place */
  pVTFWin->SysView  = pmSysView;
  pVTFWin->ParentWin= pmParentWin;
  pVTFWin->CBOK     = pmCBOK;
  pVTFWin->CBCancel = pmCBCancel;

  /* Build the window */
  pVTFWin->Window = gtk_window_new(GTK_WINDOW_DIALOG);
  pVTFWin->VBox   = gtk_vbox_new(FALSE, TRUE);
  gtk_container_add(GTK_CONTAINER(pVTFWin->Window), pVTFWin->VBox);

  /* Build the Trace information Label and pack it */
  sprintf(lString, "Trace extents:\n\nStart time: %.0lf\nEnd time: %.0lf", 
	  pmEGraph->StartTime, pmEGraph->StartTime + pmEGraph->Duration);
  pVTFWin->TraceTimeLabel = gtk_label_new(lString);
  gtk_box_pack_start(GTK_BOX(pVTFWin->VBox), pVTFWin->TraceTimeLabel, TRUE, TRUE, 5);

  /* Allocate HBoxes to put the rest of the widgets and pack them */
  pVTFWin->StartHBox  = gtk_hbox_new(FALSE, 0);
  pVTFWin->EndHBox    = gtk_hbox_new(FALSE, 0);
  pVTFWin->ButtonHBox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(pVTFWin->VBox), pVTFWin->StartHBox, TRUE, TRUE, 5);
  gtk_box_pack_start(GTK_BOX(pVTFWin->VBox), pVTFWin->EndHBox, TRUE, TRUE, 5);
  gtk_box_pack_start(GTK_BOX(pVTFWin->VBox), pVTFWin->ButtonHBox, FALSE, TRUE, 0);

  /* Create the labels and pack them */
  pVTFWin->StartTimeLabel = gtk_label_new("Start time: ");
  pVTFWin->EndTimeLabel   = gtk_label_new("End time: ");
  gtk_widget_set_usize(pVTFWin->StartTimeLabel, 80, 20);
  gtk_widget_set_usize(pVTFWin->EndTimeLabel, 80, 20);
  gtk_box_pack_start(GTK_BOX(pVTFWin->StartHBox), pVTFWin->StartTimeLabel, FALSE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(pVTFWin->EndHBox), pVTFWin->EndTimeLabel, FALSE, TRUE, 0);

  /* Create text entries and pack them */
  pVTFWin->StartTimeEntry = gtk_entry_new();
  pVTFWin->EndTimeEntry = gtk_entry_new();
  sprintf(lString, "%.0lf", pmEGraph->DrawStartTime);
  gtk_entry_set_text(GTK_ENTRY(pVTFWin->StartTimeEntry), lString);
  sprintf(lString, "%.0lf", pmEGraph->DrawStartTime + pmEGraph->Interval);
  gtk_entry_set_text(GTK_ENTRY(pVTFWin->EndTimeEntry), lString);
  gtk_box_pack_start(GTK_BOX(pVTFWin->StartHBox), pVTFWin->StartTimeEntry, TRUE, TRUE, 5);
  gtk_box_pack_start(GTK_BOX(pVTFWin->EndHBox), pVTFWin->EndTimeEntry, TRUE, TRUE, 5);

  /* Create OK and cancel buttons and pack them */
  pVTFWin->OKButton     = gtk_button_new_with_label("OK");
  pVTFWin->CancelButton = gtk_button_new_with_label("Cancel");
  GTK_WIDGET_SET_FLAGS(pVTFWin->OKButton, GTK_CAN_DEFAULT);  /* Allow OK button to be default */
  gtk_box_pack_start(GTK_BOX(pVTFWin->ButtonHBox), pVTFWin->OKButton, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(pVTFWin->ButtonHBox), pVTFWin->CancelButton, TRUE, TRUE, 0);

  /* Give it to the caller */
  return pVTFWin;
}

/******************************************************************
 * Function :
 * Description :
 * Parameters :
 * Return values :
 * History :
 * Note :
 ******************************************************************/
void VTFDestroyViewTimeFrameWindow(viewTimeFrameWindow* pmVTFWin)
{
  /* Destroy the window */
  gtk_widget_destroy(pmVTFWin->Window);

  /* Free space used by structure */
  g_free(pmVTFWin);
}
