/*
    TraceDaemon.c : Tracing daemon for the Linux kernel.
    Copyright (C) 1999, 2000, 2001, 2002 Karim Yaghmour (karim@opersys.com).

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; version 2 of the License.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/* Necessary headers */
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/sysinfo.h>
#include <sys/mman.h>
#include <dirent.h>
#if 0 /* Check whether we still need this. K.Y. */
#include <stdint.h>
#endif

#include <errno.h>

#include <UserTrace.h>

/* Local definitions */
#include "TraceDaemon.h"

/* Tracer properties */
#define TRACE_BUF_LEN           50000
#define TRACE_DATA_AREA_SIZE  1000000

/* If we use libc5 this macro is not defined. Thus, we define it here. */
/* K.Y. This was added by A. Heppel for SYSGO. I'll leave it here, but must
confess that anyone contemplating libc5 should actually be using uClibc. */
#ifndef MAP_FAILED
#define MAP_FAILED      ((__ptr_t) -1)
#endif

 /* Global variables */
int      gPID;               /* The daemon's PID */
int      gBusy;              /* Are we currently writting events to the output file */
int      gTracDev;           /* File descriptor to tracing driver */
int      gTracDataFile;      /* The file to which tracing data is dumped */
int      gTracDataAreaSize;  /* The size of the are used to buffer driver data */
int      gTracNBuffers;      /* The number of buffers to divide buffer area into, default 2 if -n option not specified */
char*    gTracDataArea;      /* The beginning of data area */
char*    gTracDataAreaP;     /* The primary data area */
char*    gTracDataAreaS;     /* The secondary data area */
options* gOptions;           /* The user given options */
int      gWaitingForThreads = FALSE; /* Flag indicating whether we're waiting for threads to get timeslices before we exit */

/* Event strings the user can specify at the command line */
char *EventOT[TRACE_MAX_EVENTS + 1] =
{
"START",
"SYS_ENTRY",
"SYS_EXIT",
"TRAP_ENTRY",
"TRAP_EXIT",
"IRQ_ENTRY",
"IRQ_EXIT",
"SCHED",
"KTIMER",
"SIRQ",
"PROCESS",
"FS",
"TIMER",
"MEM",
"SOCKET",
"IPC",
"NET",
"",   /* Buffer start */
"",   /* Buffer end */
"NEWEV",   /* New event */
"CSTM", /* Custom event */
"CHMSK",
#if SUPP_RTAI
"RTMNT",
"RTUMNT",
"RTGIRQE",
"RTGIRQX",
"RTOIRQE",
"RTOIRQX",
"RTTRAPE",
"RTTRAPX",
"RTSRQE",
"RTSRQX",
"RTSWTCHL",
"RTSWTCHR",
"RTSCHED",
"RTTASK",
"RTTIMER",
"RTSEM",
"RTMSG",
"RTRPC",
"RTMBX",
"RTFIFO",
"RTSHM",
"RTPOSIX",
"RTLXRT",
"RTLXRTI",
#endif /* SUPP_RTAI */
};

/******************************************************************
 * Function :
 *    CreateOptions()
 * Description :
 *    Creates and initializes an options structure.
 * Parameters :
 *    None.
 * Return values :
 *    The new options structure.
 * History :
 *    K.Y. 21/10/99 : Initial typing.
 * Note :
 ******************************************************************/
options* CreateOptions(void)
{
  options* pOptions;    /* Options to be created */

  /* Allocate space for new options */
  if((pOptions = (options*) malloc(sizeof(options))) == NULL)
    {
    printf("Unable to allocate space for options \n");
    exit(1);
    }

  /* Initialize options */
  pOptions->ConfigDefault      = TRUE;
  pOptions->SpecifyEvents      = FALSE;
  pOptions->EventMask          = 0;
  pOptions->SpecifyDetails     = FALSE;
  pOptions->DetailsMask        = 0;
  pOptions->ConfigCPUID        = FALSE;
  pOptions->ConfigPID          = FALSE;
  pOptions->PID                = 0;
  pOptions->ConfigPGRP         = FALSE;
  pOptions->PGRP               = 0;
  pOptions->ConfigGID          = FALSE;
  pOptions->GID                = 0;
  pOptions->ConfigUID          = FALSE;
  pOptions->UID                = 0;
  pOptions->ConfigSyscallDepth = TRUE;
  pOptions->SyscallDepth       = 0;
  pOptions->ConfigSyscallBound = FALSE;
  pOptions->UpperBound         = 0;
  pOptions->LowerBound         = 0;
  pOptions->ConfigTime         = FALSE;
  pOptions->Time.tv_sec        = 0;
  pOptions->Time.tv_usec       = 0;
  pOptions->DeviceFileName     = NULL;
  pOptions->TraceFileName      = NULL;
  pOptions->ProcFileName       = NULL;

  pOptions->ModifyTraceMask    = FALSE;
  pOptions->PrintTraceMask     = FALSE;

  pOptions->UseLocking         = TRUE;
  pOptions->DataAreaSize       = 0;
  pOptions->NBuffers           = 0;

  /* Give the options to the caller */
  return pOptions;
}

/*******************************************************************
 * Function :
 *    ParseOptions()
 * Description :
 *    Parses the command line arguments and sets the options
 *    structure to the right values.
 * Parameters :
 *    pmArgc, The number of command line options given to the program
 *    pmArgv, The argument vector given to the program
 *    pmOptions, The options structure to be set
 * Return values :
 *    NONE
 * History :
 *    K. Yaghmour, 21/10/99, Initial typing.
 * Note :
 *    This function will "exit()" if something is wrong with the
 *    command line options. If this is to go in graphic mode then
 *    gtk_init() will be called.
 *******************************************************************/
void ParseOptions(int pmArgc, char** pmArgv, options* pmOptions)
{
  int     i, j;        /* Generic indexes */
  int     lOptLen = 0; /* Length of option string */
  char*   pEnd;        /* End pointer for strtol */

  /* Read the command line arguments */
  for(i = 1; i < pmArgc; i++)
    {
    /* Get the option's length */
    lOptLen = strlen(pmArgv[i]);

#if 0
    /* DEBUG */
    printf("Analyzing option : %s \n", pmArgv[i]);
#endif

    /* Is this an option or a file name */
    if(pmArgv[i][0] == '-')
      {
      /* Is this long enough to really be an option */
      if(lOptLen < 2)
	{
        /* Hey Larry, Please give us something meaningfull! */
	printf("Incomplete option : %d \n", i);
	exit(1);
	}

      /* Depending on the option given */
      switch(pmArgv[i][1])
	{
	/* Default */
	case 'd':
	  pmOptions->ConfigDefault = TRUE;
	  break;

	/* Use lock-free version */
	case 'f':
	  pmOptions->UseLocking = FALSE;
	  break;

	/* Daemon buffer size */
	case 'b':
	case 'B':
	  /* Is the option given long enough */
	  if(lOptLen < 3)
	    {
	    printf("Daemon buffer size option without buffer size : %d \n", i);
	    exit(1);
	    }

	  /* Read the daemon's buffer size */
	  pmOptions->DataAreaSize = strtol(&(pmArgv[i][2]), &pEnd, 10);
	  if(*pEnd != '\0')
	    {
	    printf("Invalid buffer size `%s' \n", &pmArgv[i][2]);
	    exit(1);
	    }
	  break;

	/* Daemon number of buffers */
	case 'n':
	case 'N':
	  /* Is the option given long enough */
	  if(lOptLen < 3)
	    {
	    printf("Daemon number of buffers option without number of buffers : %d \n", i);
	    exit(1);
	    }

	  /* Read the daemon's number of buffers */
	  pmOptions->NBuffers = strtol(&(pmArgv[i][2]), &pEnd, 10);
	  if(*pEnd != '\0')
	    {
	    printf("Invalid number of buffers `%s' \n", &pmArgv[i][2]);
	    exit(1);
	    }
	  break;

	/* Specify the events who's details are to be logged */
	case 'D' :
	  /* Is the option given long enough */
	  if(lOptLen < 3)
	    {
	    printf("Specifying an event's details to monitor but no event given : %d \n", i);
	    exit(1);
	    }

	  /* We are specifying specific event */
	  pmOptions->SpecifyDetails = TRUE;

	  /* Using CORE we can include all core events at once into trace */
	  if (!strcmp("CORE", &(pmArgv[i][2])))
	    {
	    ltt_set_bit(TRACE_SYSCALL_ENTRY, &(pmOptions->DetailsMask));	
	    ltt_set_bit(TRACE_SYSCALL_EXIT, &(pmOptions->DetailsMask));	
	    ltt_set_bit(TRACE_TRAP_ENTRY, &(pmOptions->DetailsMask));	
	    ltt_set_bit(TRACE_TRAP_EXIT, &(pmOptions->DetailsMask));	
	    ltt_set_bit(TRACE_IRQ_ENTRY, &(pmOptions->DetailsMask));	
	    ltt_set_bit(TRACE_IRQ_EXIT, &(pmOptions->DetailsMask));	
	    ltt_set_bit(TRACE_SCHEDCHANGE, &(pmOptions->DetailsMask));	
	    }
	  else
	    {
	    /* Set event to be traced */
	    for(j = 0; j <= TRACE_MAX_EVENTS; j++)
	      {
	      /* Is this the option he gave us */
	      if(!strcmp(EventOT[j], &(pmArgv[i][2])))
		{
		ltt_set_bit(j, &(pmOptions->DetailsMask));

		/* Tracing custom events requires tracing of their declaration */
	        if (j == TRACE_CUSTOM)
	          ltt_set_bit(TRACE_NEW_EVENT, &(pmOptions->DetailsMask));

		break;
		}
	      }
	    }

	  /* Did he specify an unknown option */
	  if(j > TRACE_MAX_EVENTS)
	    {
	    printf("Option -D accompanied by unknown event %s \n", &(pmArgv[i][2]));
	    exit(1);
	    }
	/* Fall through to the next option -e/-E => the details option makes sense
	   only if the event is traced */
#if 0
	  break;
#endif

	/* Specify events to be monitored */
	case 'e':
	case 'E':
	  /* Is the option given long enough */
	  if(lOptLen < 3)
	    {
	    printf("Specifying an event to monitor but no event given : %d \n", i);
	    exit(1);
	    }

	  /* We are specifying specific event */
	  pmOptions->SpecifyEvents = TRUE;

	  /* Using CORE we can include all core events at once into trace */
	  if (!strcmp("CORE", &(pmArgv[i][2])))
	    {
	    ltt_set_bit(TRACE_SYSCALL_ENTRY, &(pmOptions->EventMask));	
	    ltt_set_bit(TRACE_SYSCALL_EXIT, &(pmOptions->EventMask));	
	    ltt_set_bit(TRACE_TRAP_ENTRY, &(pmOptions->EventMask));	
	    ltt_set_bit(TRACE_TRAP_EXIT, &(pmOptions->EventMask));	
	    ltt_set_bit(TRACE_IRQ_ENTRY, &(pmOptions->EventMask));	
	    ltt_set_bit(TRACE_IRQ_EXIT, &(pmOptions->EventMask));	
	    ltt_set_bit(TRACE_SCHEDCHANGE, &(pmOptions->EventMask));	
	    }
	  else
	    {
	    /* Set event to be traced */
	    for(j = 0; j <= TRACE_MAX_EVENTS; j++)
	      {
	      /* Is this the option he gave us */
	      if(!strcmp(EventOT[j], &(pmArgv[i][2])))
		{
		ltt_set_bit(j, &(pmOptions->EventMask));

		/* Tracing custom events requires tracing of their declaration */
	        if (j == TRACE_CUSTOM)
	          ltt_set_bit(TRACE_NEW_EVENT, &(pmOptions->EventMask));

		break;
		}
	      }
	    }

	  /* Did he specify an unknown option */
	  if(j > TRACE_MAX_EVENTS)
	    {
	    printf("Option -e or -E accompanied by unknown event %s \n", &(pmArgv[i][2]));
	    exit(1);
	    }
	  break;

	/* Trace the CPUID */
	case 'c':
	case 'C':
	  /* Set the required information */
	  pmOptions->ConfigCPUID = TRUE;
	  break;

	/* Trace only one PID */
	case 'P':
	  /* Is there enough to specify a PID */
	  if(lOptLen < 3)
	    {
	    printf("Option -P needs a PID \n");
	    exit(1);
	    }

	  /* Set the required information */
	  pmOptions->ConfigPID = TRUE;
	  pmOptions->PID = strtol(&(pmArgv[i][2]), &pEnd, 10);
	  if(*pEnd != '\0')
	    {
	    printf("Invalid pid `%s' \n", &pmArgv[i][2]);
	    exit(1);
	    }
	  break;

	/* Trace only one process group */
	case 'G':
	  /* Is there enough to specify a process group */
	  if(lOptLen < 3)
	    {
	    printf("Option -G needs a process group \n");
	    exit(1);
	    }

	  /* Set the required information */
	  pmOptions->ConfigPGRP = TRUE;
	  pmOptions->PGRP = strtol(&(pmArgv[i][2]), &pEnd, 10);
	  if(*pEnd != '\0')
	    {
	    printf("Invalid process group `%s' \n", &pmArgv[i][2]);
	    exit(1);
	    }
	  break;

	/* Trace only one GID */
	case 'g':
	  /* Is there enough to specify a GID */
	  if(lOptLen < 3)
	    {
	    printf("Option -g needs a GID \n");
	    exit(1);
	    }

	  /* Set the required information */
	  pmOptions->ConfigGID = TRUE;
	  pmOptions->GID = strtol(&(pmArgv[i][2]), &pEnd, 10);
	  if(*pEnd != '\0')
	    {
	    printf("Invalid GID `%s' \n", &pmArgv[i][2]);
	    exit(1);
	    }
	  break;

	/* Modify current trace mask */
	case 'm':
	case 'M':
	  pmOptions->ModifyTraceMask = TRUE;
	  break;

	/* Print current trace mask */
	case 'p':
	  pmOptions->PrintTraceMask = TRUE;
	  break;

	/* Trace only one UID */
	case 'u':
	case 'U':
	  /* Is there enough to specify a UID */
	  if(lOptLen < 3)
	    {
	    printf("Option -u or -U needs a UID \n");
	    exit(1);
	    }

	  /* Set the required information */
	  pmOptions->ConfigUID = TRUE;
	  pmOptions->UID = strtol(&(pmArgv[i][2]), &pEnd, 10);
	  if(*pEnd != '\0')
	    {
	    printf("Invalid UID `%s' \n", &pmArgv[i][2]);
	    exit(1);
	    }
	  break;

	/* Syscall properties */
	case 's':
	case 'S':
	  /* Is the option long enough to tell us more about the syscall setting */
	  if(lOptLen < 3)
	    {
	    printf("Option %d doesn't specify what syscall property to set \n", i);
	    exit(1);
	    }

	  /* Depending on the syscall properties */
	  switch(pmArgv[i][2])
	    {
	    /* Use depth */
	    case 'd':
	    case 'D':
	      /* Is the option long enough to tell us more about the syscall setting */
	      if(lOptLen < 4)
		{
		printf("Option %d doesn't specify the depth on which to fetch the EIP for a syscall \n", i);
		exit(1);
		}

	      /* We are fetching the eip by depth */
	      pmOptions->ConfigSyscallDepth = TRUE;
	      pmOptions->ConfigSyscallBound = FALSE;

	      /* Read the depth */
	      pmOptions->SyscallDepth = strtol(&(pmArgv[i][3]), &pEnd, 10);
	      if(*pEnd != '\0')
		{
		printf("Invalid depth `%s' \n", &pmArgv[i][3]);
		exit(1);
		}
	      break;

	    /* Specify lower limit */
	    case 'l':
	    case 'L':
	      /* Is the option long enough to tell us more about the syscall setting */
	      if(lOptLen < 4)
		{
		printf("Option %d doesn't specify the lower bound for fetch of the EIP for a syscall \n", i);
		exit(1);
		}

	      /* Was the upper bound set */
	      if(pmOptions->UpperBound == 0)
		{
		printf("Must specify upper bound before lower bound or upper bound is 0 \n");
		exit(1);
		}

	      /* We are using syscall bounds */
	      pmOptions->ConfigSyscallBound = TRUE;
	      pmOptions->ConfigSyscallDepth = FALSE;

	      /* Read the upper bound (read as 0xffffffff) */
	      pmOptions->LowerBound = strtoul(&(pmArgv[i][3]), (char**) NULL, 16);
	      break;

	    /* Specify upper limit */
	    case 'u':
	    case 'U':
	      /* Is the option long enough to tell us more about the syscall setting */
	      if(lOptLen < 4)
		{
		printf("Option %d doesn't specify the upper bound for fetch of the EIP for a syscall \n", i);
		exit(1);
		}

	      /* Read the upper bound (read as 0xffffffff) */
	      pmOptions->UpperBound = strtoul(&(pmArgv[i][3]), (char**) NULL, 16);
	      break;
	    
	    default:
	      /* I would really like to do this for you, but I wasn't taught how to! */
	      printf("Unknow option : -%c%c \n", pmArgv[i][1], pmArgv[i][2]);
	      exit(1);
	    }
	  break;

	/* Specify the time for which the daemon should run */
	case 't':
	case 'T':
	  /* Is the option long enough to tell us more about the time setting */
	  if(lOptLen < 3)
	    {
	    printf("Option %d doesn't specify what time property to set \n", i);
	    exit(1);
	    }

	  /* Depending on the time properties */
	  switch(pmArgv[i][2])
	    {
	    /* Set seconds */
	    case 's':
	    case 'S':
	      /* Is the option long enough to tell us more about the time setting */
	      if(lOptLen < 4)
		{
		printf("Option %d doesn't specify the number of seconds to run \n", i);
		exit(1);
		}

	      /* We are runing for a limited time only (like all good rebates ....) */
	      pmOptions->ConfigTime = TRUE;

	      /* Read the number of seconds */
	      pmOptions->Time.tv_sec = strtol(&(pmArgv[i][3]), &pEnd, 10);
	      if(*pEnd != '\0')
		{
		printf("Invalid number of seconds `%s' \n", &pmArgv[i][3]);
		exit(1);
		}
	      break;

	    /* Set microseconds */
	    case 'u':
	    case 'U':
	      /* Is the option long enough to tell us more about the time setting */
	      if(lOptLen < 4)
		{
		printf("Option %d doesn't specify the number of microseconds to run \n", i);
		exit(1);
		}

	      /* We are runing for a limited time only (like all good rebates ....) */
	      pmOptions->ConfigTime = TRUE;

	      /* Read the number of microseconds */
	      pmOptions->Time.tv_usec = strtol(&(pmArgv[i][3]), &pEnd, 10);
	      if(*pEnd != '\0')
		{
		printf("Invalid number of microseconds `%s' \n", &pmArgv[i][3]);
		exit(1);
		}
	      break;
	    
	    default:
	      /* I would really like to do this for you, but I wasn't taught how to! */
	      printf("Unknown option : -%c%c \n", pmArgv[i][1], pmArgv[i][2]);
	      exit(1);
	    }
	  break;

	default :
	  /* I would really like to do this for you, but I wasn't taught how to! */
	  printf("Unknown option : -%c \n", pmArgv[i][1]);
	  exit(1);
	}
      }
    else /* Definitely a file name */
      {
      /* Do we already have the device file name */
      if(pmOptions->DeviceFileName == NULL)
	{
	/* Copy the file name */
	pmOptions->DeviceFileName = (char*) malloc(lOptLen + 1);
	strncpy(pmOptions->DeviceFileName, pmArgv[i], lOptLen + 1);
	}
      else
	{
	/* Do we already have trace file name */
	if(pmOptions->TraceFileName == NULL)
	  {
	  /* Copy the file name */
	  pmOptions->TraceFileName = (char*) malloc(lOptLen + 1);
	  strncpy(pmOptions->TraceFileName, pmArgv[i], lOptLen + 1);
	  }
	else
	  {
	  /* Do we already have the /proc file name */
	  if(pmOptions->ProcFileName == NULL)
	    {
	    /* Copy the file name */
	    pmOptions->ProcFileName = (char*) malloc(lOptLen + 1);
	    strncpy(pmOptions->ProcFileName, pmArgv[i], lOptLen + 1);
	    }
	  else
	    {
	    /* This guy doesn't know what he's doing! Tell him about it and get outa here ... */
	    printf("Wrong number of command line arguments \n");
	    printf("Unknown : %s \n", pmArgv[i]);
	    exit(1);
	    }
	  }
	}
      }
    }

  /* Do we have the correct arguments */
  if(((pmOptions->DeviceFileName == NULL)
    ||(pmOptions->TraceFileName == NULL)
    ||(pmOptions->ProcFileName  == NULL))
   &&(pmOptions->ModifyTraceMask == FALSE)
   &&(pmOptions->PrintTraceMask == FALSE))
    {
    /* Insufficient number of arguments */
    printf("Insufficient number of command line arguments \n");
    printf("Usage: TraceDaemon [[Options]] [TraceDevice] [TraceDumpFile] [ProcInfoFile] \n");
    exit(1);
    }
}

/******************************************************************
 * Function :
 *    ApplyConfig()
 * Description :
 *    Applies the configuration to the trace device and the daemon.
 * Parameters :
 *    pmFileID, the file ID of the device to operate on.
 *    pmOptions, the given options.
 * Return values :
 *    None.
 * History :
 *    K.Y. 21/10/99 : Initial typing.
 * Note :
 *    This function will exit if something goes wrong during the
 *    configuration.
 ******************************************************************/
void ApplyConfig(int pmFileID, options* pmOptions)
{
  struct itimerval      lTimer;     /* Timer used for daemon's execution */
  int                   lBitPos;    /* For buffer size calculations */
  int                   lHighestSet = 0; /* For buffer size calculations */
  int                   lMask;      /* For buffer size calculations */
  int                   lRC;        /* ioctl return code */

  /* Do we set the tracing module to it's default configuration */
  if(pmOptions->ConfigDefault == TRUE)
    {
    /* Set the tracing module to it's default configuration */
    if(ioctl(pmFileID, TRACER_CONFIG_DEFAULT, 0))
      {
      /* Don't go any further */
      printf("TraceDaemon: Unable to set tracing module to it's default setting \n");
      exit(1);
      }
    printf("TraceDaemon: Tracing module set to default config \n");
    }

  /* Set which locking scheme to use (must be set before n buffers) */
  if(ioctl(pmFileID, TRACER_CONFIG_USE_LOCKING, pmOptions->UseLocking))
    {
    /* Don't go any further */
    printf("TraceDaemon: Lock-free tracing not supported on this platform (no cmpxchg) - use the locking scheme (i.e. don't use the -f option) \n");
    exit(1);
    }
  printf("TraceDaemon: Using the %s tracing scheme \n", pmOptions->UseLocking ? "locking" : "lock-free");

  /* Using lock-free scheme */
  if(pmOptions->UseLocking == FALSE)
    {
    /* Set the number of buffers (must be set before size of buffers) */
    if(ioctl(pmFileID, TRACER_CONFIG_N_MEMORY_BUFFERS, gTracNBuffers))
      {
      /* Don't go any further */
      printf("TraceDaemon: Unable to set number of data regions used for tracing.  Must be a power of 2 (default is 2).  The -n option cannot be used without the -f option \n");
      exit(1);
      }
    } 
  else if(gTracNBuffers != 2) /* We're using locking and -n was specified */
    {
    /* Don't go any further */
    printf("TraceDaemon: The -n option cannot be used without the -f option \n");
    exit(1);
    }
  printf("TraceDaemon: Configuring %d trace buffers \n", gTracNBuffers);

  /* Fix size of data buffers */
  if(pmOptions->UseLocking == FALSE)
    {
    /* Convert the given value to a power of two.  First, find the highest
       order bit set in the buffer size word. */
    for(lBitPos = 13; lBitPos < sizeof(gTracDataAreaSize)*8; lBitPos++)
      {
      lMask = 1L << lBitPos;
      if(gTracDataAreaSize & lMask) 
	lHighestSet = lBitPos;
      }

    /* Since gTracDataAreaSize is an int, need to exclude high bit */
    if(lHighestSet < (sizeof(gTracDataAreaSize)*8 - 1))
      gTracDataAreaSize = 1L << lHighestSet;
    else
      {
      printf("TraceDaemon: Unable to set data regions used for tracing.  The lockless scheme was being used, and the buffer size, converted to the next lowest power of 2, was too large\n");
      exit(1);
      }

    /* If new size is too small, return */
    if (gTracDataAreaSize < TRACER_LOCKLESS_MIN_BUF_SIZE)
      {
      printf("TraceDaemon: Unable to set data regions used for tracing.  The lockless scheme was being used, and the buffer size, converted to the next lowest power of 2, was too small (%d)\n", gTracDataAreaSize);
      exit(1);
      }

    /* If new size is still too large, return */
    if (gTracDataAreaSize > TRACER_LOCKLESS_MAX_BUF_SIZE)
      {
      printf("TraceDaemon: Unable to set data regions used for tracing.  The lockless scheme was being used, and the buffer size, converted to the next lowest power of 2, was too large (%d)\n", gTracDataAreaSize);
      exit(1);
      }
    }
  
  /* Set the memory area configuration */
  if((lRC = ioctl(pmFileID, TRACER_CONFIG_MEMORY_BUFFERS, gTracDataAreaSize)))
    {
    /* Don't go any further */
    if(lRC == -EBUSY)
      printf("TraceDaemon: Unable to set data regions used for tracing - the trace driver still has event writes pending from a previous trace \n");
    else if(pmOptions->UseLocking)
      printf("TraceDaemon: Unable to set data regions used for tracing \n");
    else
      printf("TraceDaemon: Unable to set data regions used for tracing.  If the -f option (lock-free scheme) is used, the  total buffer size (# buffers * buffer size) must not exceed 128M \n");
    exit(1);
    }

  /* Tell the user the size of the buffers */
  printf("TraceDaemon: Trace buffers are %d bytes \n", gTracDataAreaSize);

  /* Do we set the event mask */
  if(pmOptions->SpecifyEvents == TRUE)
    {
    /* Set the event mask */
    if(ioctl(pmFileID, TRACER_CONFIG_EVENTS, &(pmOptions->EventMask)))
      {

      /* Don't go any further */
      printf("TraceDaemon: Unable to set event mask \n");
      exit(1);
      }
    printf("TraceDaemon: Set event mask 0x%016llX \n", pmOptions->EventMask);
    }

  /* Do we set the details mask */
  if(pmOptions->SpecifyDetails == TRUE)
    {
    /* Set the details mask */
    if(ioctl(pmFileID, TRACER_CONFIG_DETAILS, &(pmOptions->DetailsMask)))
      {
      /* Don't go any further */
      printf("TraceDaemon: Unable to set details mask \n");
      exit(1);
      }
    printf("TraceDaemon: Set event details mask 0x%016llX \n", pmOptions->DetailsMask);
    }

  /* Do we set to log CPUID */
  if(pmOptions->ConfigCPUID == TRUE)
    {
    /* Set to log CPUID */
    if(ioctl(pmFileID, TRACER_CONFIG_CPUID, 0))
      {
      /* Don't go any further */
      printf("TraceDaemon: Unable to set the tracer to record CPUID \n");
      exit(1);
      }
    printf("TraceDaemon: Tracking CPUID \n");
    }

  /* Do we set to track a given PID */
  if(pmOptions->ConfigPID == TRUE)
    {
    /* Set to log PID */
    if(ioctl(pmFileID, TRACER_CONFIG_PID, pmOptions->PID))
      {
      /* Don't go any further */
      printf("TraceDaemon: Unable to set the tracer to track the given PID \n");
      exit(1);
      }
    printf("TraceDaemon: Tracking PID : %d \n", pmOptions->PID);
    }

  /* Do we set to track a given process group */
  if(pmOptions->ConfigPGRP == TRUE)
    {
    /* Set to log process group */
    if(ioctl(pmFileID, TRACER_CONFIG_PGRP, pmOptions->PGRP))
      {
      /* Don't go any further */
      printf("TraceDaemon: Unable to set the tracer to track the given process group \n");
      exit(1);
      }
    printf("TraceDaemon: Tracking process group : %d \n", pmOptions->PGRP);
    }

  /* Do we set to track a given GID */
  if(pmOptions->ConfigGID == TRUE)
    {
    /* Set to log GID */
    if(ioctl(pmFileID, TRACER_CONFIG_GID, pmOptions->GID))
      {
      /* Don't go any further */
      printf("TraceDaemon: Unable to set the tracer to track the given GID \n");
      exit(1);
      }
    printf("TraceDaemon: Tracking GID : %d \n", pmOptions->GID);
    }

  /* Do we set to track a given UID */
  if(pmOptions->ConfigUID == TRUE)
    {
    /* Set to log UID */
    if(ioctl(pmFileID, TRACER_CONFIG_UID, pmOptions->UID))
      {
      /* Don't go any further */
      printf("TraceDaemon: Unable to set the tracer to track the given UID \n");
      exit(1);
      }
    printf("TraceDaemon: Tracking UID : %d \n", pmOptions->UID);
    }

  /* Do we set the syscall eip depth fetch */
  if(pmOptions->ConfigSyscallDepth == TRUE)
    {
    /* Set to fetch eip depth */
    if(ioctl(pmFileID, TRACER_CONFIG_SYSCALL_EIP_DEPTH, pmOptions->SyscallDepth))
      {
      /* Don't go any further */
      printf("TraceDaemon: Unable to set syscall fetch depth \n");
      exit(1);
      }
    printf("TraceDaemon: Fetching eip for syscall on depth : %d \n", pmOptions->SyscallDepth);
    }

  /* Do we set the syscall eip bounds */
  if(pmOptions->ConfigSyscallBound == TRUE)
    {
    /* Set to eip according to bounds */
    if(ioctl(pmFileID, TRACER_CONFIG_SYSCALL_EIP_LOWER, pmOptions->LowerBound))
      {
      /* Don't go any further */
      printf("TraceDaemon: Unable to set syscall fetch lower bound \n");
      exit(1);
      }
    /* Set to eip according to bounds */
    if(ioctl(pmFileID, TRACER_CONFIG_SYSCALL_EIP_UPPER, pmOptions->UpperBound))
      {
      /* Don't go any further */
      printf("TraceDaemon: Unable to set syscall fetch upper bound \n");
      exit(1);
      }
    printf("TraceDaemon: Fetching eip for syscall between 0x%X and 0x%X \n",
	   pmOptions->LowerBound,
	   pmOptions->UpperBound);
    }

  /* Do we run the daemon only for a given time */
  if(pmOptions->ConfigTime == TRUE)
    {
    /* Set timer properties */
    memset(&(lTimer.it_interval), 0, sizeof(lTimer.it_interval));
    memcpy(&(lTimer.it_value), &(pmOptions->Time), sizeof(lTimer.it_value));

    /* Set an alarm to wake us up when we're done */
    if(setitimer(ITIMER_REAL, &lTimer, NULL) < 0)
      {
      /* Don't go any further */
      printf("TraceDaemon: Unable to set timer to regulate daemon's trace time \n");
      exit(1);
      }
    printf("TraceDaemon: Daemon will run for : (%ld, %ld) \n", pmOptions->Time.tv_sec, pmOptions->Time.tv_usec);
    }
}

/*******************************************************************
 * Function :
 *    ModifyEventMask()
 * Description :
 *    Modify the current event mask.
 * Parameters :
 *    pmOptions, the options structure.
 * Return values :
 *    NONE
 * History :
 *    K.Y., 15/04/2002, Initial typing.
 * Note :
 *******************************************************************/
void ModifyEventMask(options* pmOptions)
{
  /* Attach to the active tracing device */
  if(trace_attach() < 0)
    {
    printf("TraceDaemon: Unable to attach to trace device \n");
    exit(1);
    }  

  /* Set the event mask */
  if(trace_set_event_mask(pmOptions->EventMask) < 0)
    {
    printf("TraceDaemon: Unable to set new event mask \n");
    exit(1);
    }

  /* Print out the new event mask */
  printf("TraceDaemon: Set event mask 0x%016llX \n", pmOptions->EventMask);

  /* Detach from active tracing session */
  if(trace_detach() < 0)
    {
    printf("TraceDaemon: Unable to detach to trace device \n");
    exit(1);
    }  
}

/*******************************************************************
 * Function :
 *    PrintEventMask()
 * Description :
 *    Print the current event mask.
 * Parameters :
 *    NONE
 * Return values :
 *    NONE
 * History :
 *    K.Y., 15/04/2002, Initial typing.
 * Note :
 *******************************************************************/
void PrintEventMask(void)
{
  trace_event_mask    lEventMask;    /* The event mask */

  /* Attach to the active tracing device */
  if(trace_attach() < 0)
    {
    printf("TraceDaemon: Unable to attach to trace device \n");
    exit(1);
    }  

  /* Set the event mask */
  if(trace_get_event_mask(&lEventMask) < 0)
    {
    printf("TraceDaemon: Unable to get current event mask \n");
    exit(1);
    }

  /* Print out the new event mask */
  printf("TraceDaemon: Current event mask 0x%016llX \n", lEventMask);

  /* Detach from active tracing session */
  if(trace_detach() < 0)
    {
    printf("TraceDaemon: Unable to detach to trace device \n");
    exit(1);
    }  
}

/* Process buffers for the lockless scheme.  Called in response to a signal
   from the driver indicating that another buffer is possibly ready.  The
   pmForceAll parameter is used to force all buffers to be either written
   or discarded (used when the daemon signals tracing to stop).  The driver 
   ensures that we can never get lapped.  The buffers_produced member of 
   buffer control may be larger in reality than the snapshot we're looking 
   at indicates, but since it can never lap buffers_consumed we don't care - 
   we'll just pick up the new buffer(s) next time we're signaled. */
static void ProcessLocklessBuffers(struct buffer_control * pmBufferControl, int pmForceAll)
{
  uint32_t lBuffersReady, lStartBuffer, lEndBuffer;
  uint32_t lBufferIndex, i;
  uint32_t lBuffersConsumed = 0;
  char * lBufPtr;
  uint32_t lNBuffers = TRACE_MAX_BUFFER_NUMBER(pmBufferControl->bufno_bits);
  uint32_t lBufSize = TRACE_BUFFER_SIZE(pmBufferControl->offset_bits);
  int lForceFree = FALSE; /* Flag indicating we need to free a buffer */ 

  /* The number of buffers that have been completely reserved, not
     necessarily filled, though. */
  lBuffersReady = pmBufferControl->buffers_produced - pmBufferControl->buffers_consumed;

  /* This code implements a modification of the K42 scheme.  It doesn't
     force buffers to be written or discarded until it's absolutely 
     necessary (i.e. the buffer would be lapped soon). Thus we read as
     many buffers as we can starting with the one following the last 
     consumed.  In the normal case, as soon as we try to write a buffer
     that's not exactly filled, we break out of the buffer-writing loop
     and wait until the next buffer switch signal to pick it up.  
     If the daemon has filled up all the buffers, or if we detect while
     processing the buffers that the driver is currently writing to the
     last free buffer, we force a buffer free (and possibly lose that 
     buffer's) contents.  After the trace has been stopped and we've 
     waited 10ms/per thread for any sleeping threads to finish any 
     pending event writes they may have in progress, we force all 
     buffers to be either written or discarded.  The signal/ioctl buffer
     switch protocol ensures that we'll never actually be lapped, at the
     cost of lost events in the driver and an increased complexity in 
     the slow path of the driver's trace_reserve() code. */

  /* If the next buffer switch would cause a lap (and loss of events
     in the driver), we need to force the next unconsumed buffer free. */
  if(lBuffersReady >= lNBuffers - 1)
    lForceFree = TRUE; /* Need to force the next buffer free */

  /* We start with the buffer following the last one consumed */
  lStartBuffer = pmBufferControl->buffers_consumed % lNBuffers;

  /* We end with the last buffer possibly ready */
  lEndBuffer = lStartBuffer + lBuffersReady;

  for(i = lStartBuffer; i < lEndBuffer; i++) 
    {
    lBufferIndex = i % lNBuffers;
    /* If this buffer is complete */
    if(pmBufferControl->fill_count[lBufferIndex] == lBufSize) 
      {
      lBufPtr = gTracDataArea + lBufferIndex*lBufSize;

      /* Write the filled buffer to file */
      if(write(gTracDataFile, lBufPtr, lBufSize) < 0)
	{
	fprintf(stdout, "TraceDaemon: Unable to read from data space \n");
	exit(1);
	}
      lBuffersConsumed++;
      }
    else /* Buffer is incomplete. */
      {
      /* We can't wait for it any longer. Discard. */
      if(lForceFree || pmForceAll)
	{
	printf("TraceDaemon: WARNING!!!! \n");
	printf("TraceDaemon: INCOMPLETE EVENT BUFFER LOST (buffer size: %d, buffer index: %d, fill count: %d) \n",
	       lBufSize, lBufferIndex, pmBufferControl->fill_count[lBufferIndex]);
	lBuffersConsumed++; /* Discard. */
	}
      else
	/* We still have time to wait for this buffer (and any
	   others following it) to complete, so we'll stop here 
	   for now. */
	break;
      }
    /* Only a first buffer would need to be forced free. */
    lForceFree = FALSE;
    }

  /* We arent busy anymore */
  gBusy = FALSE;

  /* Tell the driver how many buffers we comitted */
  ioctl(gTracDev, TRACER_DATA_COMITTED, lBuffersConsumed);

  return;
}

/* Set a timer to wait 10 ms for each process currently running, to allow
   any sleeping threads in the process of writing events wake up and finish.
   If we weren't able to set the timer, return 1, otherwise 0. */
static int WaitForStragglers(void)
{
  struct sysinfo lSysInfo;
  struct itimerval lTimer;     /* Timer used for lockless waiting */
		
  /* We're now waiting for everyone to get a timeslice */
  gWaitingForThreads = TRUE;
  signal(SIGALRM, SigHandler);

  /* Set timer properties */
  memset(&(lTimer.it_interval), 0, sizeof(lTimer.it_interval));

  /* Wait at least until each thread has a chance to wake up and
     finish its event write. */
  if(sysinfo(&lSysInfo) == 0) 
    {
    /* Each thread gets 10,000 microseconds */
    long int lTotaluSecs = lSysInfo.procs * 10000;
    lTimer.it_value.tv_sec = lTotaluSecs/1000000;
    lTimer.it_value.tv_usec = lTotaluSecs - lTimer.it_value.tv_sec*1000000;

    /* Don't wait TOO long to exit, 10 secs max is plenty */ 
    if(lTimer.it_value.tv_sec > 10) 
      lTimer.it_value.tv_sec = 10;
    }
  else /* Couldn't get sysinfo for some reason, just wait a 
	  reasonable amount of time */
    {
    /* procs will be used in printf message - put something in */
    lSysInfo.procs = 0; 

    /* Wait a couple seconds if we don't have better info */
    lTimer.it_value.tv_sec = 2;
    lTimer.it_value.tv_usec = 0;
    }

  /* Set an alarm to wake us up when we're done */
  if(setitimer(ITIMER_REAL, &lTimer, NULL) < 0)
      return 0;
  else
    {
    printf("TraceDaemon: Daemon will wait for (%ld, %ld) to allow %u processes to finish writing lockless events\n",
	   lTimer.it_value.tv_sec, lTimer.it_value.tv_usec, lSysInfo.procs);
    return 0;
    }
}

/* Set a timer to wait 10 ms for each process currently running, to allow
   any sleeping threads in the process of writing events wake up and finish.
   If we weren't able to set the timer, return 1, otherwise 0. */
#if 0 /* DEBUG - used to delay consumption and max out the driver buffers */
static int sTestingFullBuffers = 1;

      /* DEBUG - simulate very busy daemon to test all driver buffers full */
void SimulateBusyDaemon(void)
{
  struct timeval lCurrentTime, lStartTime;
  int lWaitNSecs = 10;

  gettimeofday(&lStartTime, NULL);
  fprintf(stderr, "waiting %d secs to let all buffers fill\n", lWaitNSecs);

  /* This causes millions of events, but that's what we want */
  for(;;) 
    {
    gettimeofday(&lCurrentTime, NULL);
    if((lCurrentTime.tv_sec - lStartTime.tv_sec) >= lWaitNSecs) 
      break;
    }

  fprintf(stderr, "DONE waiting %d secs to let all buffers fill\n", lWaitNSecs);
}
#endif

/*******************************************************************
 * Function :
 *    SigHandler()
 * Description :
 *    The signal handling routine.
 * Parameters :
 *    pmSignalID, The ID of the signal that ocurred
 * Return values :
 *    NONE
 * History :
 *    K. Yaghmour, 05/99, Initial typing.
 * Note :
 *    This function will "exit()" if something is wrong with the
 *    command line options.
 *******************************************************************/
void SigHandler(int pmSignalID)
{
  int         lEventsLost;             /* Number of events lost */
  char*       lTempDataArea;           /* Temp pointer to data area */
  int         lRetValue;               /* Return val of buffer control ioctl */
  struct buffer_control lBufferControl; /* Filled in by buffer control ioctl */

  /* Depending on the received signal */
  switch(pmSignalID)
    {
    case SIGIO :

#if 0 /* DEBUG - simulate very busy daemon to test all driver buffers full */
      if(sTestingFullBuffers) 
	{
	SimulateBusyDaemon();
	sTestingFullBuffers = 0;
	}
#endif

      /* Get the current value of driver's buffer_control struct */
      lRetValue = ioctl(gTracDev, TRACER_GET_BUFFER_CONTROL, &lBufferControl);

      if(lRetValue != 0) 
	{
	fprintf(stdout, "TraceDaemon: Unable to get buffer control data \n");
	exit(1);
	}

      /* Are we currently busy */
      if(gBusy == TRUE)
	{
	if(lBufferControl.using_lockless)
	  return; /* No big deal, we can afford to miss a signal */
	else
	  {
	  fprintf(stdout, "TraceDaemon: Received SIGIO while busy ... \n");
	  goto end_of_trace;
	  }
	}
    
      /* We are busy */
      gBusy = TRUE;

      /* Are we in lockless mode ? */
      if(lBufferControl.using_lockless) 
	ProcessLocklessBuffers(&lBufferControl, FALSE);
      else /* Using locking scheme */
        {
	/* Exchange buffers */
	lTempDataArea  = gTracDataAreaP;
	gTracDataAreaP = gTracDataAreaS;
	gTracDataAreaS = lTempDataArea;
	  
	/* Write the filled data area to file */
	if(write(gTracDataFile, gTracDataAreaS, gTracDataAreaSize) < 0)
	  {
	  fprintf(stdout, "TraceDaemon: Unable to read from data space \n");
	  exit(1);
	  }

	/* We aren't busy anymore */
	gBusy = FALSE;

	/* Tell the driver that we comitted the data */
	ioctl(gTracDev, TRACER_DATA_COMITTED, 0);
	}

      /* We're done */
      return;
      break;

    default :
      fprintf(stdout, "TraceDaemon: Received unknown signal ... stopping \n");
    case SIGTERM :
    case SIGALRM :
      /* When the end of trace timer times out, we set a flag and another
	 timer in order to let all existing threads get their timeslices and
	 write any pending events.  We don't want to rely on the details of 
	 how the scheduler works, so we assume 10ms per thread will be more
	 than enough time to cover everyone.  We thus can get a SIGALRM for 
	 two reasons - end-of-trace timer times out and wait-for-threads
	 timer times out. */
      /* Get the current value of driver's buffer_control struct */
      lRetValue = ioctl(gTracDev, TRACER_GET_BUFFER_CONTROL, &lBufferControl);
      if(lRetValue != 0) 
	{
	fprintf(stdout, "TraceDaemon: Unable to get buffer control data \n");
	exit(1);
	}

      if(lBufferControl.using_lockless)
        {
	if(gWaitingForThreads == FALSE) /* This is end of trace signal */ 
	  {
	  /* Stop the driver - no new events can then be logged */
	  ioctl(gTracDev, TRACER_STOP, 0);

	  if(WaitForStragglers() != 0)
	    {
	    /* OK, then we'll take our chances and not wait.  Force all 
	       buffers to be written or discarded now. */
	    ProcessLocklessBuffers(&lBufferControl, TRUE);
	    }
	  else
	    return; /* We'll be back */ 
	  }
	else /* This is the end of our wait */
	  /* Force all buffers to be written or discarded */
	  ProcessLocklessBuffers(&lBufferControl, TRUE);
	} 
      else /* using locking scheme */ 
	/* Stop the driver */
	ioctl(gTracDev, TRACER_STOP, 0);

      /* Read the number of events lost */
      lEventsLost = ioctl(gTracDev, TRACER_GET_EVENTS_LOST, 0);

      /* Where there any events lost */
      if(lEventsLost > 0)
	{
	printf("TraceDaemon: WARNING!!!! \n");
	printf("TraceDaemon: %d EVENTS HAVE BEEN LOST \n", lEventsLost);
	printf("TraceDaemon: Check the size of the buffers \n");
	printf("TraceDaemon: Current buffer size %d \n", gTracDataAreaSize);
	}
    }

end_of_trace:
  /* Write what's in the current data area to file when in locking mode. */
  if(lBufferControl.using_lockless == FALSE)
    write(gTracDataFile, gTracDataAreaP, gTracDataAreaSize);

  /* OK we're out of here */
  close(gTracDev);
  close(gTracDataFile);
      
  /* Tell the user that we're done */
  fprintf(stdout, "\nTraceDaemon: End of tracing \n");
  _exit(0);
}

/*******************************************************************
 * Function :
 *    MapProc()
 * Description :
 *    Function maping out /proc.
 * Parameters :
 *    pmProcFile, file where the information is to be written
 * Return values :
 *    NONE
 * History :
 *    K. Yaghmour, 05/99, Initial typing.
 *******************************************************************/
void MapProc(int pmProcFile)
{
  int             i;                               /* Generic index */
  int             lIRQ;                            /* IRQ ID */
  int             lPID;                            /* Process' PID */
  int             lPPID;                           /* Parent's PID */
  int             lNbCPU;                          /* Number of CPUs */
  int             lSizeRead;                       /* Quantity of data read from file */
  int             lStatusFile;                     /* File containd process status */
  char            lFileName[TD_FILE_NAME_STRLEN];  /* A string containing a complete file name */
  char            lDataBuf[TD_STATUS_STRLEN];      /* Status file content */
  char            lName[TD_PROCES_NAME_STRLEN];    /* Name of the process */
  char*           lString;                         /* Generic string */
  DIR*            pDir;                            /* Directory pointer */
  FILE*           pIntFile;                        /* Int pile pointer */
  struct dirent*  pDirEntry;                       /* Directory entry */

  /* Open /proc */
  if((pDir = opendir("/proc")) == NULL)
    {
    printf("\nYour system doesn't support /proc \n");
    exit(1);
    }

  /* Read the directory through */
  while((pDirEntry = readdir(pDir)) != NULL)
    {
    /* Is this a directory entry describing a PID */
    if(isdigit(pDirEntry->d_name[0]))
      {
      /* Get the PID in number form */
      lPID = atoi(pDirEntry->d_name);

      /* Open the status file */
      snprintf(lFileName, TD_FILE_NAME_STRLEN, "/proc/%d/status", lPID);
      if((lStatusFile = open(lFileName, O_RDONLY, 0)) < 0)
	continue;

      /* Read a little bit */
      if((lSizeRead = read(lStatusFile, lDataBuf, TD_STATUS_STRLEN)) == 0)
	continue;

      /* Close the file */
      close(lStatusFile);
      
      /* Find Name and Parent PID */
      lString  = strstr(lDataBuf, "Name");
      sscanf(lString, "Name: %s", lName);
      lString  = strstr(lDataBuf, "PPid");
      sscanf(lString, "PPid: %d", &lPPID);

      /* Print out the precious information */
      snprintf(lDataBuf, TD_STATUS_STRLEN, "PID: %d; PPID: %d; NAME: %s\n", lPID, lPPID, lName);
      write(pmProcFile, lDataBuf, strlen(lDataBuf));
      }
    }

  /* Close the directory */
  closedir(pDir);

  /* Open the interrupts information file */
  if((pIntFile = fopen("/proc/interrupts", "r")) == NULL)
    {
    printf("open /proc/interrupts failed");
    return;
    }
  
  /* The first line is the column header, usually just CPU0, CPU1, etc.
     Count the number of CPUs based on "CPUn" occurences.  I didn't
     see anything else in /proc that would have been any easier
     and I need to be in this pseudofile anyway... */
  if(fgets(lDataBuf, 200, pIntFile) <= 0)
    { /* Error or EOF */
    printf("read /proc/interrupts failed");
    fclose(pIntFile);
    return;
    }

  /* Initialize the number of CPUs */
  lNbCPU = 0;

  /* Print header for CPU 0 */
  snprintf(lName, TD_PROCES_NAME_STRLEN, "CPU%d", lNbCPU);

  /* Find entry for CPU 0 */
  lString = strstr(lDataBuf, lName);

  /* Go through rest of string */
  while (lString)
    {
    /* Look for next CPU info */
    lNbCPU++;
    snprintf(lName, TD_PROCES_NAME_STRLEN, "CPU%d", lNbCPU);
    lString = strstr(lString, lName);
    }

  /* How many CPUs did we find */
  if (lNbCPU <= 0 || lNbCPU > 128)
    {  
    /* Missing or too weird */
    printf("Can't determine CPU count /proc/interrupts\n");
    fclose(pIntFile);
    return;
    }

  /* At the beginning of each valid IRQ line is a digit.  Scan for 
     it, skip through the CPU hit fields, skip the IRQ supplier (or 
     whatever that field is) and grab the rest.  This assumes the 
     IRQ supplier has no white space in the middle.  Outer fscanf
     stops on EOF, "NMI", "ERR", or other non-numeric chars. */

  /* Go to the next interrupt description in the file */
  while(fscanf(pIntFile, "%d:", &lIRQ) == 1)
    {
    /* Is this a valid  */
    if(lIRQ > 256)
      {
      printf("IRQ %d is out of range in /proc/interrutps\n", lIRQ);
      fclose(pIntFile);
      return;
      }

    /* For all CPUs */
    for(i = 0; i < lNbCPU; i++)
      /* Read one int per CPU */
      if(fscanf(pIntFile, "%s", lDataBuf) != 1)
	{
	printf("Bad INT count field seen in /proc/interrutps\n");
	fclose(pIntFile);
	return;
	}
	    
    /* The trailing space is crucial in the next format because
       it gets rid of all whitespace after the supplier field.
       That sets it up for the final fgets() */
     
    /* Read a whitespace */
    if(fscanf(pIntFile, "%s ", lDataBuf) != 1)
      {
      printf("Bad INT source field seen in /proc/interrutps\n");
      fclose(pIntFile);
      return;
      }

    /* Get the rest of the line */
    fgets(lName, 100, pIntFile);

    /* Format string describing the IRQ */
    snprintf(lDataBuf, TD_STATUS_STRLEN, "IRQ: %d; NAME: %s", lIRQ, lName);

    /* Write string to proc file */
    write(pmProcFile, lDataBuf, strlen(lDataBuf));
    }

  /* Close the interrupt information file */
  fclose(pIntFile);
}

/*******************************************************************
 * Function :
 *    main()
 * Description :
 *    The daemon entry point.
 * Parameters :
 *    pmArgc, the number of command line arguments.
 *    pmArgv, the argument vector.
 * Return values :
 *    NONE (even if declared as int).
 * History :
 *    K. Yaghmour, 05/99, Initial typing.
 *******************************************************************/
int main(int pmArgc, char** pmArgv)
{
  int              lPID;       /* The pid of the daemon */
  int              lProcFile;  /* File to which current process information is dumped */
  struct sigaction lSigAct;    /* Structure passed on to sigaction() */
  int              lTotalDataAreaSize; /* Total buffer memory in driver */

  /* Initialize the busy flag */
  gBusy = FALSE;

  /* Allocate space for the options */
  gOptions = CreateOptions();

  /* Parse the given options */
  ParseOptions(pmArgc, pmArgv, gOptions);

  /* Initialize buffer size and count to defaults or specified values */
  if(gOptions->UseLocking == TRUE)
    {
    if(gOptions->DataAreaSize == 0) 
      gTracDataAreaSize = TRACE_DATA_AREA_SIZE;
    else
      gTracDataAreaSize = gOptions->DataAreaSize;

    if(gOptions->NBuffers == 0) 
      gTracNBuffers = 2;
    else
      gTracNBuffers = gOptions->NBuffers;
    }
  else /* Using lock-free */
    {
    if(gOptions->DataAreaSize == 0) 
      gTracDataAreaSize = TRACER_LOCKLESS_DEFAULT_BUF_SIZE;
    else
      gTracDataAreaSize = gOptions->DataAreaSize;

    if(gOptions->NBuffers == 0) 
      gTracNBuffers = TRACER_LOCKLESS_DEFAULT_N_BUFFERS;
    else
      gTracNBuffers = gOptions->NBuffers;
    }

  /* Do we only need to change trace mask? */
  if(gOptions->ModifyTraceMask == TRUE)
    {
    /* Only change the event mask */
    ModifyEventMask(gOptions);

    /* Leave immediately */
    _exit(0);
    }

  /* Do we only need to print the current trace mask? */
  if(gOptions->PrintTraceMask == TRUE)
    {
    /* Only change the event mask */
    PrintEventMask();

    /* Leave immediately */
    _exit(0);
    }

  /* Become daemon */
  if((lPID = fork()) < 0)
    {
    /* Print an error message */
    printf("\nUnable to become daemon \n");

    /* Exit */
    exit(1);
    }
  else if(lPID)
    /* The parent dies */
    exit(0);

  /* Become session leader */
  gPID = setsid();

  /* Initialize structure passed to sigaction() */
  lSigAct.sa_handler = SigHandler;          /* Signal hanlder */
  sigfillset(&(lSigAct.sa_mask));           /* Signal mask */
  lSigAct.sa_flags = 0;                     /* Action flags (No re-register or unblock) */

  /* Set up the signal handler */
  if(sigaction(SIGIO, &lSigAct, NULL)) exit(1);
  if(sigaction(SIGTERM, &lSigAct, NULL)) exit(1);
  if(sigaction(SIGALRM, &lSigAct, NULL)) exit(1);

  /* Open the file to which the tracing data should be written */
  if((gTracDataFile = open(gOptions->TraceFileName, O_CREAT | O_RDWR | O_TRUNC, S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH)) < 0)
    {
    /* File ain't good */
    printf("\nUnable to open output file (errno = %d) \n", errno);
    exit(1);
    }  

  /* The output file is ready */
  printf("TraceDaemon: Output file ready \n");

  /* Open the tracing driver */
  if((gTracDev = open(gOptions->DeviceFileName, O_RDONLY, 0)) < 0)
    {
    /* Device ain't good */
    printf("\nTraceDaemon: Unable to open specified device : %s \n", gOptions->DeviceFileName);
    close(gTracDataFile);
    exit(1);
    }

  /* We opened the driver */
  printf("TraceDaemon: Trace driver open \n");

  /* Apply the configuration */
  ApplyConfig(gTracDev, gOptions);

  /* Buffers have different sizes depending on the locking mode */
  if(gOptions->UseLocking == TRUE) 
    /* The locking version always and only uses 2 'buffers' */
    lTotalDataAreaSize = gTracDataAreaSize << 1;
  else
    /* The lockless version uses n 'buffers' */
    lTotalDataAreaSize = gTracDataAreaSize * gTracNBuffers;

  /* Map the driver's buffers in our address space */
  gTracDataAreaP = mmap(NULL,
			lTotalDataAreaSize,
			PROT_READ | PROT_WRITE,
			MAP_PRIVATE,
			gTracDev,
			0);

  /* Did the mapping work out */
  if(gTracDataAreaP == MAP_FAILED)
    {
    printf("TraceDaemon: Unable to map driver's data area in daemon's space \n");
    close(gTracDataFile);
    close(gTracDev);
    exit(1);
    }

  /* For lockless processing, we need a fixed pointer to buffers. */ 
  gTracDataArea = gTracDataAreaP;

  /* Set the secondary data area */
  gTracDataAreaS = gTracDataAreaP + gTracDataAreaSize;

#if 0
  printf("TraceDaemon: Primary data area is at   => 0x%08X \n", gTracDataAreaP);
  printf("TraceDaemon: Secondary data area is at => 0x%08X \n", gTracDataAreaS);
#endif

#if 0
  gTracDataAreaP[0] = 0xF;
#endif

  /* Fire up the tracing */
  if(ioctl(gTracDev, TRACER_START, 0) < 0)
    {
    /* Don't go any further */
    printf("TraceDaemon: Unable to start the tracing \n");
    exit(1);
    }

#if 0
  if(gTracDataAreaP[0] == 0)
    {
    printf("TraceDaemon: tracing isn't functionning \n");
    exit(1);
    }
  printf("TraceDaemon: First byte is %d \n", gTracDataAreaP[0]);
#endif

  /* Map out /proc into user given file */
  if((lProcFile = creat(gOptions->ProcFileName, S_IRUSR| S_IWUSR | S_IRGRP | S_IROTH)) < 0)
    {
    /* File ain't good */
    printf("\nUnable to open output file to dump current process information \n");
    exit(1);
    }

  /* Map out /proc */
  MapProc(lProcFile);
  
  /* Tell the user that mapping /proc is done */
  printf("TraceDaemon: Done mapping /proc \n");
  
  /* Close the output file */
  close(lProcFile);

  /* Sleep forever */
  while(1)
    pause();

  /* SHOULD NEVER GET HERE! */
  exit(1);
}

